<?php
/**
 * Admin
 *
 * usage:
 * require_once plugin_dir_path( __FILE__ ) . 'includes/admin.php';
 * IMG_Processor_Admin::init();
 *
 * @package           tenandtwo-wp-plugins
 * @subpackage        tenandtwo-img-processor
 * @author            Ten & Two Systems
 * @copyright         2024 Ten & Two Systems
 */
defined( 'ABSPATH' ) or die( 'Not for browsing' );

/**
 * IMG_Processor_Admin
 * All class methods static and hooked.
 */
class IMG_Processor_Admin
{

    /**
     * init
     */
    public static function init()
    {
//if (WP_DEBUG) { trigger_error(__METHOD__, E_USER_NOTICE); }

        load_plugin_textdomain( 'tenandtwo-img-processor', false, IMG_PROCESSOR_DIR.'/languages' );

        add_action( 'admin_menu', array('IMG_Processor_Admin', 'register_pages') );
        add_action( 'admin_init', array('IMG_Processor_Admin', 'register_settings') );
        add_action( 'admin_enqueue_scripts', array('IMG_Processor_Admin', 'register_styles') );

        $filter_name = 'plugin_action_links_' . IMG_PROCESSOR_NAME."/".IMG_PROCESSOR_NAME.".php";
        add_filter( $filter_name, array('IMG_Processor_Admin', 'render_action_links') );
    }


    /**
     * return validated options array
     *  sc_img_process  boolean
     *  api_img_process boolean
     *  cli_img_process boolean
     *  search_path     string, filepath \n filepath \n filepath ...
     *  placeholder     string, image filepath
     *  cache_path      string, filepath
     *  output_type     string, [gif|jpg|png|webp|..]
     *  output_dpi      integer, [0|72|..]
     *  expire_type     string, [a|c|m][time|min]
     *  expire_value    integer
     *
     *  transform defaults  mixed
     */
    public static function validate_options( $input )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('input'),true), E_USER_NOTICE); }
if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r($_POST,true), E_USER_NOTICE); }

		$valid_expire_type = array('atime','amin','ctime','cmin');

        $options = array(
            'sc_img_process'    => 0
            , 'api_img_process' => 0
            , 'cli_img_process' => 0

            , 'search_path'     => ''
		    , 'placeholder'     => IMG_PROCESSOR_PLACEHOLDER

			, 'cache_path'      => IMG_PROCESSOR_CACHE
            , 'output_type'     => ''   // gif, jpg, png, webp
            , 'output_dpi'      => 0    // 72
			, 'expire_type'     => 'atime'
			, 'expire_value'    => 2

            , 'width'   => 500  // fit|fill|resize|crop|zoom
            , 'height'  => 300
            , 'top'     => 20   // crop|trim|border
            , 'left'    => 20
            , 'bottom'  => 20
            , 'right'   => 20
            , 'axis'    => 'y'  // flip
            , 'degrees' => 270  // rotate
            , 'percent' => 100  // scale|zoom
            , 'focus'   => 'x'  // zoom

            , 'color_r' => 0    // rotate, border
            , 'color_g' => 0
            , 'color_b' => 0
            , 'color_a' => 0

            //, 'brightness'  => 20   // brightness
            //, 'contrast'    => 20   // contrast
            //, 'smooth'      => 20   // smooth
            //, 'block_size'  => 3    // pixelate
            //, 'advanced'    => true
            //, 'subtraction' => 3    // scatter
            //, 'addition'    => 5
            );

        if (empty($_POST['reset']))
        {
            $options['sc_img_process']  = !empty($input['sc_img_process'])  ? 1 : 0;
            $options['api_img_process'] = !empty($input['api_img_process']) ? 1 : 0;
            $options['cli_img_process'] = !empty($input['cli_img_process']) ? 1 : 0;

            if (!empty($input['cache_path']))
            {
                if (is_dir($input['cache_path']) && is_writeable($input['cache_path'])) {
                    $options['cache_path'] = rtrim($input['cache_path'],'/');
                } elseif (is_dir($input['cache_path'])) {
                    $options['cache_path'] = rtrim($input['cache_path'],'/');
                    $options['sc_img_process'] = $options['api_img_process'] = $options['cli_img_process'] = 0;

                    $msg = '<strong>'.esc_html__( 'Error:  ', 'tenandtwo-img-processor' ).'</strong>'
                         . sprintf(esc_html__( "Cache Directory '%s' not writeable", 'tenandtwo-img-processor' ), $input['cache_path']);
                    IMG_Processor_Notice::error( $msg );
                } else {
                    $msg = '<strong>'.esc_html__( 'Error:  ', 'tenandtwo-img-processor' ).'</strong>'
                         . sprintf(esc_html__( "Cache Directory '%s' not found", 'tenandtwo-img-processor' ), $input['cache_path']);
                    IMG_Processor_Notice::error( $msg );
                }
            }

            if (isset($input['output_type']))
            {
                $allowed = array('gif','jpg','png','webp');
                $options['output_type'] = (in_array($input['output_type'],$allowed)) ? $input['output_type'] : '';
            }
            if (isset($input['output_dpi']))
            {
                $options['output_dpi'] = intval($input['output_dpi']);
            }

            if (!empty($input['expire_type']) && in_array($input['expire_type'], $valid_expire_type))
            {
                $options['expire_type'] = $input['expire_type'];
            }
            if (isset($input['expire_value']) && 0 <= intval($input['expire_value']))
            {
                $options['expire_value'] = intval($input['expire_value']);
            }

            if (!empty($input['search_path']))
            {
                $options['search_path'] = join("\n", IMG_Processor_Util::getRealPaths( $input['search_path'] ));
            }

            if (!empty($input['placeholder']))
            {
                $search_paths = explode("\n", $options['search_path'] ?? '');
                $search_paths[] = ABSPATH;
                $path = IMG_Processor_Util::getFileExistsLocal( $input['placeholder'], $search_paths );

                if (!empty($path)) {
                    $options['placeholder'] = $path;
                } else {
                    $options['sc_img_process'] = $options['api_img_process'] = $options['cli_img_process'] = 0;

                    $msg = '<strong>'.esc_html__( 'Error:  ', 'tenandtwo-img-processor' ).'</strong>'
                         . sprintf(esc_html__( "the Placeholder Image '%s' was NOT found", 'tenandtwo-img-processor' ), $input['placeholder']);
                    IMG_Processor_Notice::error( $msg );
                }
            }
        }

        if (!IMG_PROCESSOR_GD && ($options['sc_img_process'] || $options['api_img_process'] || $options['cli_img_process']))
        {
            $options['sc_img_process'] = $options['api_img_process'] = $options['cli_img_process'] = 0;
            $msg = '<strong>'.esc_html__( 'Error:  ', 'tenandtwo-img-processor' ).'</strong>'
                 . esc_html__( "GD extension not found", 'tenandtwo-img-processor' );
            IMG_Processor_Notice::error( $msg );
        }

        // admin_notice for changes
        self::options_update_notice( $options );

        // remove cache files
        if (!empty($_POST['cache_delete']) || !empty($_POST['cache_clean']))
        {
            $params = array(
                'cache_path'   => $options['cache_path'],
                'expire_value' => $options['expire_value'],
                'expire_type'  => (!empty($_POST['cache_delete'])) ? 'all' : $options['expire_type'],
                );
            if (empty($GD_MGR)) { $GD_MGR = new IMG_Processor_GD(); }
            $cache = $GD_MGR->cleanCache( $params );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('cache'),true), E_USER_NOTICE); }

            $msg = ((!empty($_POST['cache_delete']))  ? 'All' : 'Expired')
                . ' image files removed from '.$cache['cache_path'];
            IMG_Processor_Notice::success( $msg );
        }

if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('options'),true), E_USER_NOTICE); }
        return $options;
    }

    /**
     * create admin_notice for changes
     *  after    array
     */
    public static function options_update_notice( $after )
    {
        $before = get_option( IMG_PROCESSOR_OPTS, array() );
        $diffs = array_diff_assoc($after,$before);
        if (empty($diffs)) { return; }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('before','after','diffs'),true), E_USER_NOTICE); }

        $labels = array(
            'sc_img_process'    => esc_html__( 'Activate Shortcodes', 'tenandtwo-img-processor' ),
            'api_img_process'   => esc_html__( 'Activate API', 'tenandtwo-img-processor' ),
            'cli_img_process'   => esc_html__( 'Activate CLI', 'tenandtwo-img-processor' ),

            'search_path'       => esc_html__( 'Local Image Search Paths', 'tenandtwo-img-processor' ),
            'placeholder'       => esc_html__( 'Placeholder Image', 'tenandtwo-img-processor' ),
            'cache_path'        => esc_html__( 'Cache Directory', 'tenandtwo-img-processor' ),
            'output_type'       => esc_html__( 'Output Image Type', 'tenandtwo-img-processor' ),
            'output_dpi'        => esc_html__( 'Output Image DPI', 'tenandtwo-img-processor' ),
            'expire_type'       => esc_html__( 'Expire Type', 'tenandtwo-img-processor' ),
            'expire_value'      => esc_html__( 'Expire Value', 'tenandtwo-img-processor' ),

            'percent'   => esc_html__( 'Default Scale', 'tenandtwo-img-processor' ),
            'width'     => esc_html__( 'Default Width', 'tenandtwo-img-processor' ),
            'height'    => esc_html__( 'Default Height', 'tenandtwo-img-processor' ),
            'top'       => esc_html__( 'Default Top', 'tenandtwo-img-processor' ),
            'left'      => esc_html__( 'Default Left', 'tenandtwo-img-processor' ),
            'bottom'    => esc_html__( 'Default Bottom', 'tenandtwo-img-processor' ),
            'right'     => esc_html__( 'Default Right', 'tenandtwo-img-processor' ),
            'axis'      => esc_html__( 'Default Axis', 'tenandtwo-img-processor' ),
            'degrees'   => esc_html__( 'Default Degrees', 'tenandtwo-img-processor' ),
            'color_r'   => esc_html__( 'Default Red', 'tenandtwo-img-processor' ),
            'color_g'   => esc_html__( 'Default Green', 'tenandtwo-img-processor' ),
            'color_b'   => esc_html__( 'Default Blue', 'tenandtwo-img-processor' ),
            'color_a'   => esc_html__( 'Default Opacity', 'tenandtwo-img-processor' ),
            );

        $msg = "";
        foreach( $diffs as $key => $val )
        {
            $label = $labels[$key] ?? $key;
            $pre   = $before[$key] ?? "unset";
            $post  = $after[$key]  ?? "unset";
            if (in_array($key,array('sc_img_process','api_img_process','cli_img_process')))
            {
                $pre  = ($pre == 1)  ? 'TRUE' : 'FALSE';
                $post = ($post == 1) ? 'TRUE' : 'FALSE';
            }
            if ($key == 'search_path')
            {
                $pre  = "'".str_replace("\n"," ",$pre)."'";
                $post = "'".str_replace("\n"," ",$post)."'";
            }
            if ($msg) { $msg .= "<br/>"; }
            $msg .= $label . esc_html(" : $pre => $post");
        }
        if ($msg) {
            $msg = '<strong>'.esc_html__( 'IMG Processor Settings updated', 'tenandtwo-img-processor' ).':</strong><br/>'.$msg;
            IMG_Processor_Notice::success( $msg );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('diffs','msg'),true), E_USER_NOTICE); }
        }

    }


    /**
     * register_pages
     * @uses add_options_page( string $page_title, string $menu_title, string $capability, string $menu_slug, callable $function = '', int $position = null )
     */
    public static function register_pages()
    {
        add_options_page(
            'settings.php',
            'IMG Processor',
            'manage_options',                 // capability
            'img_processor_settings',         // page_name
            array('IMG_Processor_Admin','render_page_settings'),
            //1,                              // menu position
            );
    }

    /**
     * render_page_settings
     * @uses settings_fields( string $option_group )
     * @uses do_settings_sections( string $page )
     */
    public static function render_page_settings()
    {
        echo '<div class="wrap">';
        echo '<h1>'.esc_html(_x( 'IMG Processor Settings', 'settings page title', 'tenandtwo-img-processor' )).'</h1>';
        echo '<form action="options.php" method="post">';

        echo '<hr size="1" />';
        settings_fields( 'img_processor_settings' );       // option_group
        do_settings_sections( 'img_processor_settings' );  // page_name

        echo '<hr size="1" />';
        echo '<table class="form-table" role="presentation"><tbody><tr><td>';
        submit_button();
        echo '</td><td align="right">';
        submit_button( 'Restore Default Settings', 'secondary', 'reset', false, array('reset'=>1) );
        echo '</td></tr></tbody></table>';

        echo '</form>';
        echo '</div>';

//if (WP_DEBUG) { print_r( get_option( IMG_PROCESSOR_OPTS, array() )); }
    }


    /**
     * register_settings
     * @uses register_setting( string $option_group, string $option_name, array $args = array() )
     * @uses add_settings_section( string $id, string $title, callable $callback, string $page, array $args = array() )
     * @uses add_settings_field( string $id, string $title, callable $callback, string $page, string $section = 'default', array $args = array() )
     */
    public static function register_settings()
    {
        // register setting group and options array
        register_setting(
            'img_processor_settings',
            IMG_PROCESSOR_OPTS,
            array(
                'type' => 'array',
                'sanitize_callback' => array('IMG_Processor_Admin','validate_options'),
                'default' => array(),
            ));

        // add section 'main' to settings page
        add_settings_section(
            'img_processor_settings_main',
            '', //esc_html(_x( 'Main', 'section title', 'tenandtwo-img-processor' ))
            array('IMG_Processor_Admin','render_section_main'),
            'img_processor_settings',
            array(
//                 'before_section' => '<p>before</p>',
//                 'after_section'  => '<p>after</p>',
//                 'section_class'  => 'img_processor_settings',
            ));

        // add field 'sc_img_process'
        add_settings_field(
            'img_processor_sc_img_process',
            esc_html(_x( 'Activate Shortcodes', 'field_label', 'tenandtwo-img-processor' )),
            array('IMG_Processor_Admin','render_setting_sc_img_process'),
            'img_processor_settings',
            'img_processor_settings_main',
            array(
                'label_for' => 'img_processor_sc_img_process',  // wrap title in label
                //'class'  => 'classname',                      // add to tr
            ));

        // add field 'api_img_process'
        add_settings_field(
            'img_processor_api_img_process',
            esc_html(_x( 'Activate API', 'field_label', 'tenandtwo-img-processor' )),
            array('IMG_Processor_Admin','render_setting_api_img_process'),
            'img_processor_settings',
            'img_processor_settings_main',
            array(
                'label_for' => 'img_processor_api_img_process',
            ));

        // add field 'cli_img_process'
        add_settings_field(
            'img_processor_cli_img_process',
            esc_html(_x( 'Activate CLI', 'field_label', 'tenandtwo-img-processor' )),
            array('IMG_Processor_Admin','render_setting_cli_img_process'),
            'img_processor_settings',
            'img_processor_settings_main',
            array(
                'label_for' => 'img_processor_cli_img_process',
            ));

        // add field 'search_path'
        add_settings_field(
            'img_processor_search_path',
            esc_html(_x( 'Local Image Search Paths', 'field_label', 'tenandtwo-img-processor' )),
            array('IMG_Processor_Admin','render_setting_search_path'),
            'img_processor_settings',
            'img_processor_settings_main',
            array(
                'label_for' => 'img_processor_search_path',
            ));

        // add field 'placeholder'
        add_settings_field(
            'img_processor_placeholder',
            esc_html(_x( 'Placeholder Image', 'field_label', 'tenandtwo-img-processor' )),
            array('IMG_Processor_Admin','render_setting_placeholder'),
            'img_processor_settings',
            'img_processor_settings_main',
            array(
                'label_for' => 'img_processor_placeholder',
            ));

        // add field 'cache_path'
        add_settings_field(
            'img_processor_cache_path',
            esc_html(_x( 'Cache Directory', 'field_label', 'tenandtwo-img-processor' )),
            array('IMG_Processor_Admin','render_setting_cache_path'),
            'img_processor_settings',
            'img_processor_settings_main',
            array(
                'label_for' => 'img_processor_cache_path',
            ));

        // add field 'expire_type'
        // add field 'expire_value'
        add_settings_field(
            'img_processor_expire_type',
            esc_html(_x( 'Cache Lifetime', 'field_label', 'tenandtwo-img-processor' )),
            array('IMG_Processor_Admin','render_setting_expire_type'),
            'img_processor_settings',
            'img_processor_settings_main',
            array(
                'label_for' => 'img_processor_expire_type',
            ));

        // add field 'output_type'
        add_settings_field(
            'img_processor_output_type',
            esc_html(_x( 'Cache File Type', 'field_label', 'tenandtwo-img-processor' )),
            array('IMG_Processor_Admin','render_setting_output_type'),
            'img_processor_settings',
            'img_processor_settings_main',
            array(
                'label_for' => 'img_processor_output_type',
            ));

        // add field 'output_dpi'
        add_settings_field(
            'img_processor_output_dpi',
            esc_html(_x( 'Cache File Resolution', 'field_label', 'tenandtwo-img-processor' )),
            array('IMG_Processor_Admin','render_setting_output_dpi'),
            'img_processor_settings',
            'img_processor_settings_main',
            array(
                'label_for' => 'img_processor_output_dpi',
            ));

    }

    /**
     * render section header
     */
    public static function render_section_main()
    {
        echo 'PHP\'s <a href="https://www.php.net/manual/en/book.image.php" target="_blank">'.esc_html__( 'GD extension', 'tenandtwo-img-processor' ).'</a> ';
        if (!IMG_PROCESSOR_GD) {
            esc_html_e( 'is NOT available', 'tenandtwo-img-processor' );
        } else {
            esc_html_e( 'is available', 'tenandtwo-img-processor' );
            echo '&nbsp;:&nbsp;';
            echo esc_html(
                'GD v'.GD_VERSION
                );
        }

        echo '<br/>';

        echo 'PHP\'s <a href="https://www.php.net/manual/en/book.imagick.php" target="_blank">'.esc_html__( 'Imagick extension', 'tenandtwo-img-processor' ).'</a> ';
        if (!IMG_PROCESSOR_IMAGICK) {
            esc_html_e( 'is NOT available', 'tenandtwo-img-processor' );
        } else {
            global $IMG_MGR;
            if (empty($IMG_MGR)) { $IMG_MGR = new IMG_Processor_Imagick(); }
            $version = ($IMG_MGR->initHandler()) ? $IMG_MGR->imagick_info['version'] : '';

            esc_html_e( 'is available', 'tenandtwo-img-processor' );
            echo '&nbsp;:&nbsp;';
            echo esc_html(
                'Imagick v'.$version
                );
        }
    }

    /**
     * render settings field: sc_img_process
     */
    public static function render_setting_sc_img_process()
    {
        $options = get_option( IMG_PROCESSOR_OPTS, array() );
        $value = !empty($options['sc_img_process']);

        echo '<input type="checkbox"'
            . ' id="img_processor_sc_img_process" name="'.esc_attr(IMG_PROCESSOR_OPTS.'[sc_img_process]').'"'
            . ' value="1"'
            . ((defined( 'GD_VERSION' ) && $value) ? ' checked': '')
            . ' />';

        $html = '<strong>[img_process transform="</strong>{type}<strong>" output="</strong>{info|img|html|figure}<strong>"]</strong>{source}<strong>[/img_process]</strong>';
//        $html .= __( ' - Transform and cache Images', 'tenandtwo-img-processor' );
        $html .= '<ul>';
        $html .= '<li><code><strong>[img_info]</strong>{source}<strong>[/img_info]</strong></code></li>';
        $html .= '<li><code><strong>[img_fit width="</strong>{int}<strong>" height="</strong>{int}<strong>"]</strong>{source}<strong>[/img_fit]</strong></code></li>';
        $html .= '<li><code><strong>[img_fill width="</strong>{int}<strong>" height="</strong>{int}<strong>"]</strong>{source}<strong>[/img_fill]</strong></code></li>';
        $html .= '<li><code><strong>[img_resize width="</strong>{int}<strong>" height="</strong>{int}<strong>"]</strong>{source}<strong>[/img_resize]</strong></code></li>';
        $html .= '<li><code><strong>[img_crop top="</strong>{int}<strong>" left="</strong>{int}<strong>" width="</strong>{int}<strong>" height="</strong>{int}<strong>"]</strong>{source}<strong>[/img_crop]</strong></code></li>';
        $html .= '<li><code><strong>[img_trim top="</strong>{int}<strong>" left="</strong>{int}<strong>" bottom="</strong>{int}<strong>" right="</strong>{int}<strong>"]</strong>{source}<strong>[/img_trim]</strong></code></li>';
        $html .= '<li><code><strong>[img_border top="</strong>{int}<strong>" left="</strong>{int}<strong>" bottom="</strong>{int}<strong>" right="</strong>{int}<strong>"]</strong>{source}<strong>[/img_border]</strong></code></li>';
        $html .= '<li><code><strong>[img_flip axis="</strong>{x|y}<strong>"]</strong>{source}<strong>[/img_flip]</strong></code></li>';
        $html .= '<li><code><strong>[img_rotate degrees="</strong>{int}<strong>"]</strong>{source}<strong>[/img_rotate]</strong></code></li>';
        $html .= '<li><code><strong>[img_scale percent="</strong>{int}<strong>"]</strong>{source}<strong>[/img_scale]</strong></code></li>';
        $html .= '<li><code><strong>[img_zoom percent="</strong>{int}<strong>" focus="</strong>{X|N|S|E|W|NE|NW|SE|SW}<strong>" width="</strong>{int}<strong>" height="</strong>{int}<strong>"]</strong>{source}<strong>[/img_zoom]</strong></code></li>';

//         if (IMG_PROCESSOR_IMAGICK && false) {
//             $html .= '<li><code><strong>[img_colorize color_r="</strong>{int}<strong>" color_g="</strong>{int}<strong>" color_b="</strong>{int}<strong>" color_a="</strong>{int}<strong>"]</strong>{source}<strong>[/img_colorize]</strong></code></li>';
//             $html .= '<li><code><strong>[img_grayscale]</strong>{source}<strong>[/img_grayscale]</strong></code></li>';
//             $html .= '<li><code><strong>[img_negate]</strong>{source}<strong>[/img_negate]</strong></code></li>';
//         }

        $html .= '<li><a href="'.IMG_PROCESSOR_DOCS.'" target="_blank">'.esc_html__( 'View all options', 'tenandtwo-img-processor' ).'</a> <span class="dashicons dashicons-external"></span></li>';
        $html .= '</ul>';
        echo wp_kses($html, 'post');
    }

    /**
     * render settings field: api_img_process
     */
    public static function render_setting_api_img_process()
    {
        $options = get_option( IMG_PROCESSOR_OPTS, array() );
        $value = !empty($options['api_img_process']);

        echo '<input type="checkbox"'
            . ' id="img_processor_api_img_process" name="'.esc_attr(IMG_PROCESSOR_OPTS.'[api_img_process]').'"'
            . ' value="1"'
            . ((defined( 'GD_VERSION' ) && $value) ? ' checked': '')
            . ' />';

        $html = '<strong>&lt;img src="'.IMG_PROCESSOR_URL.'?transform=</strong>{type}<strong>&source=</strong>{source}<strong>&output_nocache=</strong>{0|1}<strong>" />"</strong>';
        $html .= '<ul>';
//        $html .= '<li><code><strong>[img_info]</strong>{source}<strong>[/img_info]</strong></code></li>';

        $html .= '<li><a href="'.IMG_PROCESSOR_DOCS.'" target="_blank">'.esc_html__( 'View all options', 'tenandtwo-img-processor' ).'</a> <span class="dashicons dashicons-external"></span></li>';
        $html .= '</ul>';
        echo wp_kses($html, 'post');
    }

    /**
     * render settings field: cli_img_process
     */
    public static function render_setting_cli_img_process()
    {
        $options = get_option( IMG_PROCESSOR_OPTS, array() );
        $value = !empty($options['cli_img_process']);

        echo '<input type="checkbox"'
            . ' id="img_processor_cli_img_process" name="'.esc_attr(IMG_PROCESSOR_OPTS.'[cli_img_process]').'"'
            . ' value="1"'
            . ((defined( 'GD_VERSION' ) && $value) ? ' checked': '')
            . ' />';

        $html = '<strong>wp img_process &lt;</strong>{subcommand}<strong>></strong>';
        $html .= '<ul>';
        $html .= '<li><code><strong>wp img_process &lt;</strong>{transform-type}<strong>> &lt;</strong>{source}<strong>> --output="</strong>{info|file|..}<strong>"</strong></code></li>';
        $html .= '<li><code><strong>wp img_process cache_clean</strong></code></li>';
        $html .= '<li><code><strong>wp img_process cache_delete</strong></code></li>';
        $html .= '<li><code><strong>wp img_process version [--types]</strong></code></li>';

        $html .= '<li><a href="'.IMG_PROCESSOR_DOCS.'" target="_blank">'.esc_html__( 'View all options', 'tenandtwo-img-processor' ).'</a> <span class="dashicons dashicons-external"></span></li>';
        $html .= '</ul>';
        echo wp_kses($html, 'post');
    }

    /**
     * render settings field: search_path
     */
    public static function render_setting_search_path()
    {
        $options = get_option( IMG_PROCESSOR_OPTS, array() );
        $value = $options['search_path'] ?? '';

        echo '<textarea rows="4" cols="80"'
            . ' id="img_processor_search_path" name="'.esc_attr(IMG_PROCESSOR_OPTS.'[search_path]').'"'
            . '>' . esc_textarea($value)
            . '</textarea>';

        $html = '<p>';
        $html .= __( '- Specify local directories containing source image files. ', 'tenandtwo-img-processor' );
        $html .= '<br/>';
        $html .= __( '- The default path', 'tenandtwo-img-processor' )
            . ' <code>'.ABSPATH.'</code> '
            . __( 'will be searched last. ', 'tenandtwo-img-processor' );
        $html .= '</p>';
        echo wp_kses($html, 'post');
    }

    /**
     * render settings field: placeholder
     */
    public static function render_setting_placeholder()
    {
        $options = get_option( IMG_PROCESSOR_OPTS, array() );
        $value = $options['placeholder'] ?? '';

        echo '<input type="text" size="80"'
            . ' id="img_processor_placeholder" name="'.esc_attr(IMG_PROCESSOR_OPTS.'[placeholder]').'"'
            . ' value="'.esc_attr($value).'"'
            . ' placeholder="'.esc_attr(IMG_PROCESSOR_PLACEHOLDER).'"'
            . ' />';

        $html = '<p>';
        $html .= __( '- Specify a default image to display when the requested image is not available. ', 'tenandtwo-img-processor' );
        $html .= '</p>';

        if (!empty($options['placeholder']) && !is_file($options['placeholder'])) {
            $html .= '<p>- <strong>'.__( 'ERROR:  ', 'tenandtwo-img-processor' ).'</strong>';
            $html .= __( 'Placeholder Image NOT found', 'tenandtwo-img-processor' );
            $html .= '</p>';
        }

        echo wp_kses($html, 'post');
    }

    /**
     * render settings field: cache_path
     */
    public static function render_setting_cache_path()
    {
        $options = get_option( IMG_PROCESSOR_OPTS, array() );
        $value = $options['cache_path'] ?? '';

        echo '<input type="text" size="80"'
            . ' id="img_processor_cache_path" name="'.esc_attr(IMG_PROCESSOR_OPTS.'[cache_path]').'"'
            . ' value="'.esc_attr($value).'"'
            . ' placeholder="'.esc_attr(IMG_PROCESSOR_CACHE).'"'
            . ' />';

        $html = '<p>';
        $html .= __( '- Specify a public directory for storing transformed images. ', 'tenandtwo-img-processor' );
        $html .= '</p>';

        $html .= '<p>';
        $html .= __( '- It is best practice to separate the cache from the plugin.', 'tenandtwo-img-processor' );
        $html .= __( '  For example, use ', 'tenandtwo-img-processor' );
        $html .= '<code><strong>'.WP_CONTENT_DIR.'/image-cache</strong></code>';
        $html .= '</p>';

        if (!empty($options['cache_path']))
        {
            if (!is_dir($options['cache_path'])) {
                $html .= '<p>- <strong>'.__( 'ERROR:  ', 'tenandtwo-img-processor' ).'</strong>';
                $html .= __( 'Cache Directory NOT found.', 'tenandtwo-img-processor' );
                $html .= '</p>';
            } elseif (!is_writeable($options['cache_path'])) {
                $html .= '<p>- <strong>'.__( 'ERROR:  ', 'tenandtwo-img-processor' ).'</strong>';
                $html .= __( 'Cache Directory NOT writeable.', 'tenandtwo-img-processor' );
                $html .= '</p>';
            }
        }

        echo wp_kses($html, 'post');
    }

    /**
     * render settings field: expire_type, expire_value
     */
    public static function render_setting_expire_type()
    {
        $options = get_option( IMG_PROCESSOR_OPTS, array() );
        $expire_type  = $options['expire_type']  ?? 'atime';
        $expire_value = $options['expire_value'] ?? '2';

        echo '<table class="setting-columns"><tr><td>';

        echo '<input type="text" size="4"'
            . ' id="img_processor_expire_value" name="'.esc_attr(IMG_PROCESSOR_OPTS.'[expire_value]').'"'
            . ' value="'.esc_attr($expire_value).'"'
            . ' placeholder="'.esc_attr($expire_value).'"'
            . ' />';

        echo '<select id="img_processor_expire_type" name="'.esc_attr(IMG_PROCESSOR_OPTS.'[expire_type]').'">'
            . '<option value="atime"'.(($expire_type == "atime") ? ' selected': '').'>Days from Last Access</option>'
            . '<option value="amin"' .(($expire_type == "amin")  ? ' selected': '').'>Minutes from Last Access</option>'
//             . '<option value="ctime"'.(($expire_type == "ctime") ? ' selected': '').'>Days from Creation</option>'
//             . '<option value="cmin"' .(($expire_type == "cmin")  ? ' selected': '').'>Minutes from Creation</option>'
            . '</select>';

        echo '</td><td align="right">';
        submit_button( _x( 'Delete Expired Files Now', 'button text', 'tenandtwo-img-processor' ),
            'secondary', 'cache_clean', false, array('cache_clean'=>1) );
        echo '</td><td align="left">';
        submit_button( _x( 'Delete All Files Now', 'button text', 'tenandtwo-img-processor' ),
            'secondary', 'cache_delete', false, array('cache_delete'=>1) );
        echo '</td></tr></table>';

        $html = '<p>';
        $html .= __( '- Expired files are removed periodically following PHP\'s <code>session.gc_probability</code>.', 'tenandtwo-img-processor' );
        $html .= '</p>';
        echo wp_kses($html, 'post');
    }

    /**
     * render settings field: output_type
     */
    public static function render_setting_output_type()
    {
        $options = get_option( IMG_PROCESSOR_OPTS, array() );
        $output_type = $options['output_type'] ?? '';

        echo '<select id="img_processor_output_type" name="'.esc_attr(IMG_PROCESSOR_OPTS.'[output_type]').'">'
            . '<option value="none">Same As Input</option>'
            . '<option value="gif"' .(($output_type == "gif")  ? ' selected': '').'>GIF</option>'
            . '<option value="jpg"' .(($output_type == "jpg")  ? ' selected': '').'>JPG</option>'
            . '<option value="png"' .(($output_type == "png")  ? ' selected': '').'>PNG</option>'
            . '<option value="webp"'.(($output_type == "webp") ? ' selected': '').'>WebP</option>'
            . '</select>';

        $html = '<p>';
        $html .= __( ' - Specify the default type for processed images.', 'tenandtwo-img-processor' );
//         $html .= '<br/>';
        $html .= __( '  To override, use', 'tenandtwo-xslt-processor' )
            . ' <code><strong>output_type="</strong>{gif|jpg|png|webp}<strong>"</strong></code> ';
        $html .= '</p>';
        echo wp_kses($html, 'post');
    }

    /**
     * render settings field: output_dpi
     */
    public static function render_setting_output_dpi()
    {
        $options = get_option( IMG_PROCESSOR_OPTS, array() );
        $output_dpi  = $options['output_dpi']  ?? '0';

        echo '<select id="img_processor_output_dpi" name="'.esc_attr(IMG_PROCESSOR_OPTS.'[output_dpi]').'">'
            . '<option value="0">Same As Input</option>'
            . '<option value="72"'.(($output_dpi == "72") ? ' selected': '').'>72 DPI</option>'
            . '</select>';

        $html = '<p>';
        $html .= __( ' - Specify the default resolution for processed images.', 'tenandtwo-img-processor' );
//         $html .= '<br/>';
        $html .= __( '  To override, use', 'tenandtwo-xslt-processor' )
            . ' <code><strong>output_dpi="</strong>{0|72|..}<strong>"</strong></code> ';
        $html .= '</p>';
        echo wp_kses($html, 'post');
    }


    /**
     * add css for admin
     */
    public static function register_styles()
    {
        wp_enqueue_style( 'img-proc-admin', plugin_dir_url(__FILE__) . 'css/img-proc-admin.css' );
    }

    /**
     * render_action_links below plugin name
     */
    public static function render_action_links( $actions )
    {
        $links = array(
            '<a href="'. esc_url( get_admin_url(null, 'options-general.php?page=img_processor_settings') ) .'">'
                . esc_html(_x( 'Settings', 'plugins action link', 'tenandtwo-img-processor' ))
                . '</a>'
            );
        return array_merge( $links, $actions );
    }


}  // end IMG_Processor_Admin
