<?php
/**
 * Returns an image data (git, jpg, png, webp) with optional transformations.
 *
 * SAMPLE - transform remote
 *  /wp-content/plugins/tenandtwo-img-processor/api.php?percent=60&source=https://a.b.c/image-source/atest.jpeg
 * SAMPLE - transform local
 *  /wp-content/plugins/tenandtwo-img-processor/api.php?axis=x&source=/wp-content/uploads/image-source/atest.gif
 *
 * SAMPLE - clean cache
 *  /wp-content/plugins/tenandtwo-img-processor/api.php?cache=clean
 * SAMPLE - delete cache
 *  /wp-content/plugins/tenandtwo-img-processor/api.php?cache=delete
 *
 *
 * transform=resize
 * The image will be resized and stretched if "width" and "height" are passed as params.
 * The "width", "height", and "transform=resize" parameters are all required.  If any one
 * of those parameters is missing, then "resize" behaves like "fit".
 * For instance, if width and height are set to "100" and "500" respectively,
 * the image will be output at 100x500.
 * SAMPLE: <img src="/image.php?source=/test.jpg&transform=resize&width=100&height=500" />
 *
 * transform=fit
 * The image will be resized proportionally if "width" or "height" are passed as params.
 * If both are provided, the image will be sized to fit the defined rectangle.
 * For instance, if the original image is square and then the width and
 * height are set to "100" and "500" respectively, the image will be output at 100x100.
 * SAMPLE: <img src="/image.php?source=/test.jpg&width=100&height=500" />
 *
 * transform=fill
 * The image will be resized proportionally and cropped to fill a defined rectagle.
 * The "width", "height", and "transform=fill" parameters are all required.  If any one
 * of those parameters is missing, then "fill" behaves like "fit".
 * SAMPLE: <img src="/image.php?source=/test.jpg&transform=fill&width=200&height=200" />
 *
 * transform=crop
 * The transform=crop param is required.  A portion of the image is returned,
 * starting at the "top","left" position, and extending "width" and/or "height" pixels.
 * For instance, a square from the top left of an image.
 * SAMPLE: <img src="/image.php?transform=crop&source=/test.jpg&top=10&left=10&width=100&height=100" />
 *
 * transform=trim
 * The image will be trimmed if "top", "left", "bottom" or "right" are passed as params.
 * For instance, if top is set to "10" then the top 10 pixels are removed.
 * SAMPLE: <img src="/image.php?source=/test.jpg&top=10" />
 *
 * transform=border
 * The transform=border param is required.  A colored border will be added to the image
 * as specified in the "top", "left", "bottom" and "right".  The border color is specified
 * in the "color_r", "color_g" and "color_b" params.
 * For instance, if top, left, bottom and right are set to "10", and color_r is set to 255,
 * then the image is surrounded by 10 pixel wide red border.
 * SAMPLE: <img src="/image.php?transform=border&source=/test.jpg&top=10&left=10&bottom=10&right=10&top=255" />
 *
 * transform=flip
 * The image will be flipped along the x-axis or y-axis if "axis" is passed as a param.
 * For instance, if axis is set to "y" the image is flipped vertically.
 * SAMPLE: <img src="/image.php?source=/test.jpg&flip=y" />
 *
 * transform=rotate
 * The image will be rotated if "degrees" is passed as a param.  Triangles created by
 * the rotation, if any, can be filled with "color_r", "color_g", and "color_b".
 * For instance, if degrees is set to "270" the image is rotated 90 degrees counter-clockwise.
 * SAMPLE: <img src="/image.php?source=/test.jpg&degrees=270" />
 *
 * transform=scale
 * The image will be scaled if "percent" is passed as a param.
 * For instance, if percent is set to "50" and the original image is 640x480 then
 * the image will be output at 320x240.
 * SAMPLE: <img src="/image.php?source=/test.jpg&percent=50" />
 *
 * transform=zoom
 * The transform=zoom param is required.  The image is scaled by "percent", then
 * cropped to the requestd "width" x "height".  The default output dimensions are the
 * same as the input.  The `focus` parameter specifies one of nine (9) positions.
 * The default crops the result from the center of the image.  The values 'NE|NW|SE|SW'
 * select the corners; the values 'N|S|E|W' select the sides.
 * SAMPLE: <img src="/image.php?transform=zoom&source=/test.jpg&percent=150&focus=NW&width=200&height=200" />
 *
 */
defined( 'ABSPATH' ) or die( 'Not for browsing' );

if (!defined('DEBUG')) { define('DEBUG', !empty($_REQUEST['debug'])); }
$starttime = IMG_Processor_Util::getMicrotime();
//trigger_error("TEST USER ERROR", E_USER_ERROR);

if (empty($GD_MGR)) { $GD_MGR = new IMG_Processor_GD(); }
if (empty($GD_MGR)) { throw new Exception('ERROR : missing GD_MGR'); }

//$IMG_DEFAULTS array - see tenandtwo-img-processor/api.php

/* ------------------------------ */
// DELETE CACHE FILES

if (!empty($_REQUEST['cache'])) {
    $params = array(
        "expire_type" => ($_REQUEST['cache'] == "delete") ? "all" : null
    );
    if (!$GD_MGR->cleanCache( $params ))
        { throw new Exception("ERROR : cleanCache(".$_REQUEST['cache'].") error"); }
}

/* ------------------------------ */
// SOURCE FILE PARAM

$file = $_REQUEST['source'] ?? null;

if (filter_var($file, FILTER_VALIDATE_URL)) {
    $file = IMG_Processor_Util::getFileExistsRemote( $file );
} else {
    $options = get_option( IMG_PROCESSOR_OPTS, array() );
    $search_paths = explode("\n", $options['search_path'] ?? '');
    $search_paths[] = ABSPATH;
    $file = IMG_Processor_Util::getFileExistsLocal( $file, $search_paths );
}

if (empty($file)) {
    $placeholder = (isset($_REQUEST['placeholder'])) ? ABSPATH.$_REQUEST['placeholder'] : $IMG_DEFAULTS['placeholder'];
    $file = (is_file($placeholder)) ? realpath($placeholder) : '404';
}

// if (basename($file) == '_blank.gif')
//     { $transform = "none"; }


/* ------------------------------ */
// OUTPUT PARAMS

$output_file    = (!empty($_REQUEST['output_file']))   ? $_REQUEST['output_file'] : $IMG_DEFAULTS['output_file'];
$output_type    = (!empty($_REQUEST['output_type']))   ? $_REQUEST['output_type'] : $IMG_DEFAULTS['output_type'];
$output_dpi     = (!empty($_REQUEST['output_dpi']))    ? $_REQUEST['output_dpi']  : $IMG_DEFAULTS['output_dpi'];
$output_nocache = (isset($_REQUEST['output_nocache'])) ? !empty($_REQUEST['output_nocache']) : $IMG_DEFAULTS['output_nocache'];


/* ------------------------------ */
// TRANSFORM PARAMS

// transform as passed
$transform = $_REQUEST['transform'] ?? null;

// transform implied
while (empty($transform)) {
    if (!empty($_REQUEST['percent']))
        { $transform = "scale"; break; }
    //if (!empty($_REQUEST['width']) && !empty($_REQUEST['height']))  // both
    //    { $transform = "fill"; break; }
    if (!(empty($_REQUEST['width']) && empty($_REQUEST['height'])))   // either
        { $transform = "fit"; break; }
    if (!(empty($_REQUEST['top']) && empty($_REQUEST['left']) && empty($_REQUEST['bottom']) && empty($_REQUEST['right'])))
        { $transform = "trim"; break; }
    if (!empty($_REQUEST['axis']))
        { $transform = "flip"; break; }
    if (!empty($_REQUEST['degrees']))
        { $transform = "rotate"; break; }
    break;
}
// transform configed
if (empty($transform))
    { $transform = $IMG_DEFAULTS['transform'] ?? null; }

$width      = null;  // fit|fill|resize|crop|zoom
$height     = null;
$top        = null;  // crop|trim|border
$left       = null;
$bottom     = null;
$right      = null;
$axis       = null;  // flip
$degrees    = null;  // rotate
$percent    = null;  // scale|zoom
$focus      = null;  // zoom
$color_r    = null;  // colorize
$color_g    = null;
$color_b    = null;
//$color_a    = null;

switch ($transform) {
    case "fit":
    case "fill":
    case "resize":
        if (!empty($_REQUEST['width']))  { $width  = $_REQUEST['width']; }
        if (!empty($_REQUEST['height'])) { $height = $_REQUEST['height']; }
        if (empty($width) && empty($height)) {
            $width  = $IMG_DEFAULTS['width']  ?? 0;
            $height = $IMG_DEFAULTS['height'] ?? 0;
        }
        break;
    case "crop":
        if (!empty($_REQUEST['top']))    { $top = $_REQUEST['top']; }
        if (!empty($_REQUEST['left']))   { $left = $_REQUEST['left']; }
        if (!empty($_REQUEST['width']))  { $width  = $_REQUEST['width']; }
        if (!empty($_REQUEST['height'])) { $height = $_REQUEST['height']; }
        if (empty($top) && empty($left) && empty($bottom) && empty($right)) {
            $top    = $IMG_DEFAULTS['top']    ?? 0;
            $left   = $IMG_DEFAULTS['left']   ?? 0;
            $width  = $IMG_DEFAULTS['width']  ?? 0;
            $height = $IMG_DEFAULTS['height'] ?? 0;
        }
        break;
    case "border":
        if (isset($_REQUEST['color_r']))  { $color_r = $_REQUEST['color_r']; }
        if (isset($_REQUEST['color_g']))  { $color_g = $_REQUEST['color_g']; }
        if (isset($_REQUEST['color_b']))  { $color_b = $_REQUEST['color_b']; }
        //if (isset($_REQUEST['color_a']))  { $color_a = $_REQUEST['color_a']; }
        //break;  // continue to "trim"
    case "trim":
        if (!empty($_REQUEST['top']))    { $top = $_REQUEST['top']; }
        if (!empty($_REQUEST['left']))   { $left = $_REQUEST['left']; }
        if (!empty($_REQUEST['bottom'])) { $bottom = $_REQUEST['bottom']; }
        if (!empty($_REQUEST['right']))  { $right = $_REQUEST['right']; }
        if (empty($top) && empty($left) && empty($bottom) && empty($right)) {
            $top    = $IMG_DEFAULTS['top']    ?? 0;
            $left   = $IMG_DEFAULTS['left']   ?? 0;
            $bottom = $IMG_DEFAULTS['bottom'] ?? 0;
            $right  = $IMG_DEFAULTS['right']  ?? 0;
        }
        break;
    case "flip":
        if (!empty($_REQUEST['axis'])) { $axis = $_REQUEST['axis']; }
        if (empty($axis) || !in_array($axis,array('x','y'))) {
            $axis  = $IMG_DEFAULTS['axis'] ?? 'y';
        }
        break;
    case "rotate":
        if (!empty($_REQUEST['degrees'])) { $degrees = $_REQUEST['degrees']; }
        if (isset($_REQUEST['color_r']))  { $color_r = $_REQUEST['color_r']; }
        if (isset($_REQUEST['color_g']))  { $color_g = $_REQUEST['color_g']; }
        if (isset($_REQUEST['color_b']))  { $color_b = $_REQUEST['color_b']; }
        //if (isset($_REQUEST['color_a']))  { $color_a = $_REQUEST['color_a']; }   // 0.0 to 1.0 = transparent to opaque
        if (empty($degrees)) {
            $degrees = $IMG_DEFAULTS['degrees'] ?? 90;
        }
        break;
    case "scale":
        if (!empty($_REQUEST['percent'])) { $percent = $_REQUEST['percent']; }
        if (empty($percent)) {
            $percent = $IMG_DEFAULTS['percent'] ?? 100;
        }
        break;
    case "zoom":
        if (!empty($_REQUEST['percent'])) { $percent = $_REQUEST['percent']; }
        if (!empty($_REQUEST['focus']))   { $focus   = $_REQUEST['focus']; }
        if (!empty($_REQUEST['width']))   { $width   = $_REQUEST['width']; }
        if (!empty($_REQUEST['height']))  { $height  = $_REQUEST['height']; }
        if (empty($percent)) {
            $percent = $IMG_DEFAULTS['percent'] ?? 100;
        }
        break;
    default:
        break;
}


/* ------------------------------ */
// GET/CREATE CACHE

$params = $result = false;
if ($file != '404') {
    $params = array(
        "input" => array(
            "file"         => $file
            )
        , "output" => array(
            "file"          => $output_file
            , "type"        => $output_type
            , "dpi"         => $output_dpi
            , "nocache"     => $output_nocache
            )
        , "transform" => array(
            "type"          => $transform
            , "width"       => $width       // fit|fill|resize|crop|zoom
            , "height"      => $height
            , "top"         => $top         // crop|trim|border
            , "left"        => $left
            , "bottom"      => $bottom
            , "right"       => $right
            , "axis"        => $axis        // flip
            , "degrees"     => $degrees     // rotate
            , "percent"     => $percent     // scale|zoom
            , "focus"       => $focus       // zoom
            , "color_r"     => $color_r     // colorize
            , "color_g"     => $color_g
            , "color_b"     => $color_b
            //, "color_a"     => $color_a
            )
        );
    $result = $GD_MGR->getImageCache( $params );
}


/* ------------------------------ */
// OUTPUT :

if (DEBUG) {  // DEBUG
    if (!empty($result['data'])) { $result['data'] = strlen($result['data']) . " bytes"; }

    echo '<hr size="1">';
    echo "GD_MGR->gd_info: <pre>".print_r($GD_MGR->gd_info,true)."</pre>";
    echo "_REQUEST: <pre>".print_r($_REQUEST,true)."</pre>";
    echo "params: <pre>".print_r($params,true)."</pre>";
    echo "result: <pre>".htmlentities(print_r($result,true))."</pre>";

    echo "runtime : ".sprintf("%.4f",(IMG_Processor_Util::getMicrotime() - $starttime)) . " seconds<br/>\n";
    echo "memory usage: ".size_format( memory_get_usage(true), 2 )."<br/>\n";
    echo "peak usage: ".size_format( memory_get_peak_usage(true), 2 )."<br/>\n";
    exit();
}

if (!empty($result['file'])) {
    header("Content-Type: " . $result['content_type']);
    header("Content-Transfer-Encoding: binary");
    header("Content-Length: " . filesize($result['file']));
    readfile($result['file']);
    exit();
}

if (!empty($result['data'])) {
    header("Content-Type: " . $result['content_type']);
    header("Content-Transfer-Encoding: binary");
    header("Content-Length: " . $result['bytes']);
    echo $result['data'];
    exit();
}

/* ------------------------------ */

header($_SERVER["SERVER_PROTOCOL"]." 404 Not Found");
?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <title>404 Not Found</title>
</head>
<body>
    <h1>Not Found</h1>
    <p>The requested image was not found on this server.</p>
    <p><?=$_SERVER['REQUEST_URI']?></p>
</body>
</html>
