<?php
/**
 * WP-CLI commands
 *
 * see https://make.wordpress.org/cli/
 *
 * @package           tenandtwo-wp-plugins
 * @subpackage        tenandtwo-img-processor
 * @author            Ten & Two Systems
 * @copyright         2023 Ten & Two Systems
 */

defined( 'ABSPATH' ) or die( 'Not for browsing' );

/**
 * iff CLI for IMG Processor disabled
 */
function img_process_not_active( $args ) {
    WP_CLI::error( "The 'img_process' command is not active. To enable it, see IMG Processor Settings > Activate CLI." );
}

class IMG_Processor_CLI
{
    /**
     * init : register self as WP_CLI command
     */
    public static function init()
    {
        if (!(defined( 'WP_CLI' ) && WP_CLI)) { return; }
//if (WP_DEBUG) { trigger_error(__METHOD__, E_USER_NOTICE); }

        $options = get_option( IMG_PROCESSOR_OPTS, array() );
        if (empty($options['cli_img_process']))
        {
            WP_CLI::add_command( 'img_process', 'img_process_not_active' );
            return;
        }

        WP_CLI::add_command( 'img_process', 'IMG_Processor_CLI' );
    }


	/**
	 * Returns 'Hello <name>!'
	 *
     * ## OPTIONS
     *
     * <name>
     * : The name of the person to greet.
     *
	 * [--color_r=<color_r>]
     * : test param
     *
     * ## EXAMPLES
     *
     *   # Say hello, single word.
     *   $ wp --allow-root img_process hello World
     *   Success: Hello World!
     *
     *   # Say hello, multiple words.
     *   $ wp --allow-root img_process hello "Great Big World"
     *   Success: Hello Great Big World!
     *
     * @when after_wp_load
	 */
    public function hello( $args, $assoc_args )
    {
if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('args','assoc_args'),true), E_USER_NOTICE); }
// 	    $name = $args[0] ?? 'World';
// 		WP_CLI::success( "Hello $name!" );

        $test = array_diff(array('color_r','color_g','color_b','color_a'), array_keys($assoc_args));
		WP_CLI::line( print_r($test,true) );

// wp --allow-root option patch insert img_processor_options color_a 0
// wp --allow-root option patch update img_processor_options color_a 1
// wp --allow-root option patch delete img_processor_options color_a

	}


	/**
	 * Return current plugin options.
	 *
     * ## EXAMPLES
     *
     *   wp --allow-root img_process settings
     *
     * @when after_wp_load
	 */
    public function settings( $args, $assoc_args )
    {
        $result = array(IMG_PROCESSOR_OPTS => get_option( IMG_PROCESSOR_OPTS, array() ));
		WP_CLI::line( print_r($result,true) );
	}


	/**
	 * Return GD and Imagick extension info.
	 *
     * ## OPTIONS
     *
     * [--types]
     * : Include listing of accepted image types.
	 *
     * ## EXAMPLES
     *
     *   wp --allow-root img_process version
     *   wp --allow-root img_process version --types
     *
     * @when after_wp_load
	 */
    public function version( $args, $assoc_args )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('args','assoc_args'),true), E_USER_NOTICE); }

        $result = array(
            'gd'      => 'extension not available',
            'imagick' => 'extension not available',
            );
        if (IMG_PROCESSOR_GD)
        {
            global $GD_MGR;
            if (empty($GD_MGR)) { $GD_MGR = new IMG_Processor_GD(); }
            $GD_MGR->initHandler();
            $result['gd'] = $GD_MGR->gd_info;

            $result['gd']['input'] = (!empty($assoc_args['types']))
                ? $GD_MGR->gd_image_input
                : "use '--types' for supported image formats";
            $result['gd']['output'] = (!empty($assoc_args['types']))
                ? $GD_MGR->gd_image_output
                : "use '--types' for supported image formats";
        }
        if (IMG_PROCESSOR_IMAGICK)
        {

            global $IMG_MGR;
            if (empty($IMG_MGR)) { $IMG_MGR = new IMG_Processor_Imagick(); }
            $IMG_MGR->initHandler();
            $result['imagick'] = $IMG_MGR->imagick_info;

            $result['imagick']['types'] = (!empty($assoc_args['types']))
                ? $IMG_MGR->imagick_types
                : "use '--types' for supported image formats";
        }
		WP_CLI::line( print_r($result,true) );
	}


	/**
	 * Delete expired cache items.
	 *
     * ## EXAMPLES
     *
     *   wp --allow-root img_process cache_clean
     *
     * @when after_wp_load
	 */
    public function cache_clean( $args, $assoc_args )
    {
        $options = get_option( IMG_PROCESSOR_OPTS, array() );
        $params = array(
            "cache_path"   => $assoc_args['cache_path']   ?? $options['cache_path'],
            "expire_type"  => $assoc_args['expire_type']  ?? $options['expire_type'],
            "expire_value" => $assoc_args['expire_value'] ?? $options['expire_value'],
            );
        if (!empty($assoc_args['cache_path'])) {
            WP_CLI::warning( print_r($params,true) );
            WP_CLI::confirm( "Delete expired files inside the '".$options['cache_path']."' directory?", $assoc_args );
        }

        if (IMG_PROCESSOR_IMAGICK)
        {
            global $IMG_MGR;
            if (empty($IMG_MGR)) { $IMG_MGR = new IMG_Processor_Imagick(); }
            $cache = $IMG_MGR->cleanCache( $params );
        }
        else
        {
            global $GD_MGR;
            if (empty($GD_MGR)) { $GD_MGR = new IMG_Processor_GD(); }
            $cache = $GD_MGR->cleanCache( $params );
        }
        WP_CLI::success( print_r($cache,true) );
	}

	/**
	 * Delete all cache items.
	 *
     * ## EXAMPLES
     *
     *   wp --allow-root img_process cache_delete
     *
     * @when after_wp_load
	 */
    public function cache_delete( $args, $assoc_args )
    {
        $options = get_option( IMG_PROCESSOR_OPTS, array() );
        $params = array(
            "cache_path"   => $assoc_args['cache_path']   ?? $options['cache_path'],
            "expire_type"  => 'all',
            );
        if (!empty($assoc_args['cache_path'])) {
            WP_CLI::warning( print_r($params,true) );
            WP_CLI::confirm( "DELETE ALL files inside the '".$options['cache_path']."' directory?", $assoc_args );
        }

        if (IMG_PROCESSOR_IMAGICK)
        {
            global $IMG_MGR;
            if (empty($IMG_MGR)) { $IMG_MGR = new IMG_Processor_Imagick(); }
            $cache = $IMG_MGR->cleanCache( $params );
        }
        else
        {
            global $GD_MGR;
            if (empty($GD_MGR)) { $GD_MGR = new IMG_Processor_GD(); }
            $cache = $GD_MGR->cleanCache( $params );
        }
		WP_CLI::success( print_r($cache,true) );
	}


	/**
	 * Get information for an image.
	 * Keys in the info array are also valid '--output' values.
	 *
     * ## OPTIONS
     *
     * <source>
     * : Source image file path or URL.
	 *
	 * [--output=<format>]
     * : Command result format : info|file|..
     * ---
     * default: info
     * options:
     *   - info
     *   - json
     *   - img
     *   - file
     *   - url
     *   - img
     *   - width
     *   - height
     *   - type
     *   - readable
     *   - writeable
     *   - bytes
     *   - datetime
     *   - content_type
     *   - dir
     *   - basename
     *   - name
     *   - ext
     *   - alt
     *   - _usage
     * ---
     *
     * ## EXAMPLES
     *
     *   wp --allow-root img_process info --output=info ./atest.gif
     *   wp --allow-root img_process info --output=file ./atest.gif
     *   wp --allow-root img_process info --output=bytes ./atest.gif
     *
     * @when after_wp_load
	 */
    public function info( $args, $assoc_args )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('args','assoc_args'),true), E_USER_NOTICE); }

        // change default output format to full info array
        if (empty($assoc_args['output'])) { $assoc_args['output'] = 'info'; }

        $transform = array(
            'type'      => 'none'
            );
        $image = $this->img_cache( $args, $assoc_args, $transform );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('args','assoc_args','transform','image'),true), E_USER_NOTICE); }
        if ($image) { $this->img_result( $args, $assoc_args, $image ); }
	}


	/**
	 * Resize image proportionally to fit requested 'width' and/or 'height'.
	 *
     * ## OPTIONS
     *
     * <source>
     * : Source image file path or URL.
	 *
	 * [--output=<format>]
     * : Command result format : info|file|..
     * ---
     * default: file
     * options:
     *   - info
     *   - json
     *   - img
     *   - file
     *   - url
     *   - img
     *   - width
     *   - height
     *   - type
     *   - readable
     *   - writeable
     *   - bytes
     *   - datetime
     *   - content_type
     *   - dir
     *   - basename
     *   - name
     *   - ext
     *   - alt
     *   - _usage
     * ---
	 *
     * [--width=<integer>]
     * : Maximum width in pixels.
	 *
     * [--height=<integer>]
     * : Maximum height in pixels.
	 *
     * [--output_type=<type>]
     * : Output image type.  'auto' = no change.
	 *
     * [--output_dpi=<integer>]
     * : Output image resolution.  '0' = no change.
	 *
     * [<output_file>]
     * : Destination image path, overrides configured cache path and image type.
	 *
     * ## EXAMPLES
     *
     *   wp --allow-root img_process fit --width=200 ./atest.gif
     *   wp --allow-root img_process fit --height=200 ./atest.gif
     *   wp --allow-root img_process fit --width=200 --height=200 ./atest.gif
     *
     * @when after_wp_load
	 */
    public function fit( $args, $assoc_args )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('args','assoc_args'),true), E_USER_NOTICE); }

        $width  = (isset($assoc_args['width']))  ? intval($assoc_args['width'])  : 0;
        $height = (isset($assoc_args['height'])) ? intval($assoc_args['height']) : 0;
        if (empty($width) && empty($height))
        {
            $options = get_option( IMG_PROCESSOR_OPTS, array() );
            $width  = $options['width']  ?? 0;
            $height = $options['height'] ?? 0;
        }

        $transform = array(
            'type'      => 'fit'
            , 'width'   => $width
            , 'height'  => $height
            );
        $image = $this->img_cache( $args, $assoc_args, $transform );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('args','assoc_args','transform','image'),true), E_USER_NOTICE); }
        if ($image) { $this->img_result( $args, $assoc_args, $image ); }
	}


	/**
	 * Resize image proportionally to fill requested 'width' and 'height'.
	 *
     * ## OPTIONS
     *
     * <source>
     * : Source image file path or URL.
	 *
	 * [--output=<format>]
     * : Command result format : info|file|..
     * ---
     * default: file
     * options:
     *   - info
     *   - json
     *   - img
     *   - file
     *   - url
     *   - img
     *   - width
     *   - height
     *   - type
     *   - readable
     *   - writeable
     *   - bytes
     *   - datetime
     *   - content_type
     *   - dir
     *   - basename
     *   - name
     *   - ext
     *   - alt
     *   - _usage
     * ---
	 *
     * --width=<integer>
     * : Width in pixels.
	 *
     * --height=<integer>
     * : Height in pixels.
	 *
     * [--output_type=<type>]
     * : Output image type.  'auto' = no change.
	 *
     * [--output_dpi=<integer>]
     * : Output image resolution.  '0' = no change.
	 *
     * [<output_file>]
     * : Destination image path, overrides configured cache path and image type.
	 *
     * ## EXAMPLES
     *
     *   wp --allow-root img_process fill --width=200 --height=200 ./atest.gif
     *
     * @when after_wp_load
	 */
    public function fill( $args, $assoc_args )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('args','assoc_args'),true), E_USER_NOTICE); }

        $width  = (isset($assoc_args['width']))  ? intval($assoc_args['width'])  : 0;
        $height = (isset($assoc_args['height'])) ? intval($assoc_args['height']) : 0;

        $transform = array(
            'type'      => 'fill'
            , 'width'   => $width
            , 'height'  => $height
            );
        $image = $this->img_cache( $args, $assoc_args, $transform );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('args','assoc_args','transform','image'),true), E_USER_NOTICE); }
        if ($image) { $this->img_result( $args, $assoc_args, $image ); }
	}


	/**
	 * Stretch image to cover requested 'width' and 'height'.
	 *
     * ## OPTIONS
     *
     * <source>
     * : Source image file path or URL.
	 *
	 * [--output=<format>]
     * : Command result format : info|file|..
     * ---
     * default: file
     * options:
     *   - info
     *   - json
     *   - img
     *   - file
     *   - url
     *   - img
     *   - width
     *   - height
     *   - type
     *   - readable
     *   - writeable
     *   - bytes
     *   - datetime
     *   - content_type
     *   - dir
     *   - basename
     *   - name
     *   - ext
     *   - alt
     *   - _usage
     * ---
	 *
     * [--width=<integer>]
     * : Maximum width in pixels.
	 *
     * [--height=<integer>]
     * : Maximum height in pixels.
	 *
     * [--output_type=<type>]
     * : Output image type.  'auto' = no change.
	 *
     * [--output_dpi=<integer>]
     * : Output image resolution.  '0' = no change.
	 *
     * [<output_file>]
     * : Destination image path, overrides configured cache path and image type.
	 *
     * ## EXAMPLES
     *
     *   wp --allow-root img_process resize --width=200 ./atest.gif
     *   wp --allow-root img_process resize --height=200 ./atest.gif
     *   wp --allow-root img_process resize --width=200 --height=200 ./atest.gif
     *
     * @when after_wp_load
	 */
    public function resize( $args, $assoc_args )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('args','assoc_args'),true), E_USER_NOTICE); }

        $width  = (isset($assoc_args['width']))  ? intval($assoc_args['width'])  : 0;
        $height = (isset($assoc_args['height'])) ? intval($assoc_args['height']) : 0;
        if (empty($width) && empty($height))
        {
            $options = get_option( IMG_PROCESSOR_OPTS, array() );
            $width  = $options['width']  ?? 0;
            $height = $options['height'] ?? 0;
        }

        $transform = array(
            'type'      => 'resize'
            , 'width'   => $width
            , 'height'  => $height
            );
        $image = $this->img_cache( $args, $assoc_args, $transform );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('args','assoc_args','transform','image'),true), E_USER_NOTICE); }
        if ($image) { $this->img_result( $args, $assoc_args, $image ); }
	}


	/**
     * Extract a (width x height) portion of the image starting at (top, left) position.
	 *
     * ## OPTIONS
     *
     * <source>
     * : Source image file path or URL.
	 *
	 * [--output=<format>]
     * : Command result format : info|file|..
     * ---
     * default: file
     * options:
     *   - info
     *   - json
     *   - img
     *   - file
     *   - url
     *   - img
     *   - width
     *   - height
     *   - type
     *   - readable
     *   - writeable
     *   - bytes
     *   - datetime
     *   - content_type
     *   - dir
     *   - basename
     *   - name
     *   - ext
     *   - alt
     *   - _usage
     * ---
	 *
     * [--top=<integer>]
     * : Starting position from top edge
	 *
     * [--left=<integer>]
     * : Starting position from left edge
	 *
     * [--width=<integer>]
     * : Maximum width.  '0' = no max.
	 *
     * [--height=<integer>]
     * : Maximum height.  '0' = no max.
	 *
     * [--output_type=<type>]
     * : Output image type.  'auto' = no change.
	 *
     * [--output_dpi=<integer>]
     * : Output image resolution.  '0' = no change.
	 *
     * [<output_file>]
     * : Destination image path, overrides configured cache path and image type.
	 *
     * ## EXAMPLES
     *
     *   wp --allow-root img_process crop --top=10 --left=10 --width=80 --height=80 ./atest.gif
     *   wp --allow-root img_process crop --top=30 --width=80 ./atest.gif
     *   wp --allow-root img_process crop --left=30 --height=80 ./atest.gif
     *
     * @when after_wp_load
	 */
    public function crop( $args, $assoc_args )
    {
if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('args','assoc_args'),true), E_USER_NOTICE); }

        $top    = (isset($assoc_args['top']))    ? intval($assoc_args['top'])    : 0;
        $left   = (isset($assoc_args['left']))   ? intval($assoc_args['left'])   : 0;
        $width  = (isset($assoc_args['width']))  ? intval($assoc_args['width'])  : 0;
        $height = (isset($assoc_args['height'])) ? intval($assoc_args['height']) : 0;
        if (empty($top) && empty($left) && empty($width) && empty($height))
        {
            $options = get_option( IMG_PROCESSOR_OPTS, array() );
            $top    = $options['top']    ?? 0;
            $left   = $options['left']   ?? 0;
            $width  = $options['width']  ?? 0;
            $height = $options['height'] ?? 0;
        }

        $transform = array(
            'type'      => 'crop'
            , 'top'     => $top
            , 'left'    => $left
            , 'width'   => $width
            , 'height'  => $height
            );
        $image = $this->img_cache( $args, $assoc_args, $transform );
if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('args','assoc_args','transform','image'),true), E_USER_NOTICE); }
        if ($image) { $this->img_result( $args, $assoc_args, $image ); }
	}


	/**
	 * Resize the image canvas by removing pixels along each edge.
	 *
     * ## OPTIONS
     *
     * <source>
     * : Source image file path or URL.
	 *
	 * [--output=<format>]
     * : Command result format : info|file|..
     * ---
     * default: file
     * options:
     *   - info
     *   - json
     *   - img
     *   - file
     *   - url
     *   - img
     *   - width
     *   - height
     *   - type
     *   - readable
     *   - writeable
     *   - bytes
     *   - datetime
     *   - content_type
     *   - dir
     *   - basename
     *   - name
     *   - ext
     *   - alt
     *   - _usage
     * ---
	 *
     * [--top=<integer>]
     * : Pixels to remove from top edge.  '0' = no change.
	 *
     * [--left=<integer>]
     * : Pixels to remove from left edge.  '0' = no change.
	 *
     * [--bottom=<integer>]
     * : Pixels to remove from bottom edge.  '0' = no change.
	 *
     * [--right=<integer>]
     * : Pixels to remove from right edge.  '0' = no change.
	 *
     * [--output_type=<type>]
     * : Output image type.  'auto' = no change.
	 *
     * [--output_dpi=<integer>]
     * : Output image resolution.  '0' = no change.
	 *
     * [<output_file>]
     * : Destination image path, overrides configured cache path and image type.
	 *
     * ## EXAMPLES
     *
     *   wp --allow-root img_process trim --top=10 --left=10 --bottom=10 --right=10 ./atest.gif
     *   wp --allow-root img_process trim --top=30 --bottom=30 ./atest.gif
     *   wp --allow-root img_process trim --left=30 --right=30 ./atest.gif
     *
     * @when after_wp_load
	 */
    public function trim( $args, $assoc_args )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('args','assoc_args'),true), E_USER_NOTICE); }

        $top    = (isset($assoc_args['top']))    ? intval($assoc_args['top'])    : 0;
        $left   = (isset($assoc_args['left']))   ? intval($assoc_args['left'])   : 0;
        $bottom = (isset($assoc_args['bottom'])) ? intval($assoc_args['bottom']) : 0;
        $right  = (isset($assoc_args['right']))  ? intval($assoc_args['right'])  : 0;
        if (empty($top) && empty($left) && empty($bottom) && empty($right))
        {
            $options = get_option( IMG_PROCESSOR_OPTS, array() );
            $top    = $options['top']    ?? 0;
            $left   = $options['left']   ?? 0;
            $bottom = $options['bottom'] ?? 0;
            $right  = $options['right']  ?? 0;
        }

        $transform = array(
            'type'      => 'trim'
            , 'top'     => $top
            , 'left'    => $left
            , 'bottom'  => $bottom
            , 'right'   => $right
            );
        $image = $this->img_cache( $args, $assoc_args, $transform );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('args','assoc_args','transform','image'),true), E_USER_NOTICE); }
        if ($image) { $this->img_result( $args, $assoc_args, $image ); }
	}


	/**
	 * Resize the image canvas by adding pixels along each edge.
	 *
     * ## OPTIONS
     *
     * <source>
     * : Source image file path or URL.
	 *
	 * [--output=<format>]
     * : Command result format : info|file|..
     * ---
     * default: file
     * options:
     *   - info
     *   - json
     *   - img
     *   - file
     *   - url
     *   - img
     *   - width
     *   - height
     *   - type
     *   - readable
     *   - writeable
     *   - bytes
     *   - datetime
     *   - content_type
     *   - dir
     *   - basename
     *   - name
     *   - ext
     *   - alt
     *   - _usage
     * ---
	 *
     * [--top=<integer>]
     * : Pixels to add to top edge.  '0' = no change.
	 *
     * [--left=<integer>]
     * : Pixels to add to left edge.  '0' = no change.
	 *
     * [--bottom=<integer>]
     * : Pixels to add to bottom edge.  '0' = no change.
	 *
     * [--right=<integer>]
     * : Pixels to add to right edge.  '0' = no change.
	 *
     * [--color_r=<integer>]
     * : Background color 0-255
	 *
     * [--color_g=<integer>]
     * : Background color 0-255
	 *
     * [--color_b=<integer>]
     * : Background color 0-255
	 *
     * [--color_a=<float>]
     * : Background opacity 0.0-1.0
	 *
     * [--output_type=<type>]
     * : Output image type.  'auto' = no change.
	 *
     * [--output_dpi=<integer>]
     * : Output image resolution.  '0' = no change.
	 *
     * [<output_file>]
     * : Destination image path, overrides configured cache path and image type.
	 *
     * ## EXAMPLES
     *
     *   wp --allow-root img_process border --top=10 --left=10 --bottom=10 --right=10 ./atest.gif
     *   wp --allow-root img_process border --top=30 --bottom=30 ./atest.gif
     *   wp --allow-root img_process border --left=30 --right=30 ./atest.gif
     *
     * @when after_wp_load
	 */
    public function border( $args, $assoc_args )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('args','assoc_args'),true), E_USER_NOTICE); }

        $top    = (isset($assoc_args['top']))    ? intval($assoc_args['top'])    : 0;
        $left   = (isset($assoc_args['left']))   ? intval($assoc_args['left'])   : 0;
        $bottom = (isset($assoc_args['bottom'])) ? intval($assoc_args['bottom']) : 0;
        $right  = (isset($assoc_args['right']))  ? intval($assoc_args['right'])  : 0;
        if (empty($top) && empty($left) && empty($bottom) && empty($right))
        {
            $options = get_option( IMG_PROCESSOR_OPTS, array() );
            $top    = $options['top']    ?? 0;
            $left   = $options['left']   ?? 0;
            $bottom = $options['bottom'] ?? 0;
            $right  = $options['right']  ?? 0;
        }

        $transform = array(
            'type'      => 'border'
            , 'top'     => $top
            , 'left'    => $left
            , 'bottom'  => $bottom
            , 'right'   => $right
            );
        // color params
        if (isset($assoc_args['color_r'])) { $transform['color_r'] = intval($assoc_args['color_r']); }
        if (isset($assoc_args['color_g'])) { $transform['color_g'] = intval($assoc_args['color_g']); }
        if (isset($assoc_args['color_b'])) { $transform['color_b'] = intval($assoc_args['color_b']); }
        if (isset($assoc_args['color_a'])) { $transform['color_a'] = floatval($assoc_args['color_a']); }

        $image = $this->img_cache( $args, $assoc_args, $transform );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('args','assoc_args','transform','image'),true), E_USER_NOTICE); }
        if ($image) { $this->img_result( $args, $assoc_args, $image ); }
	}


	/**
	 * Flip the image horizontally (x-axis) or vertically (y-axis).
	 *
     * ## OPTIONS
     *
     * <source>
     * : Source image file path or URL.
	 *
	 * [--output=<format>]
     * : Command result format : info|file|..
     * ---
     * default: file
     * options:
     *   - info
     *   - json
     *   - img
     *   - file
     *   - url
     *   - img
     *   - width
     *   - height
     *   - type
     *   - readable
     *   - writeable
     *   - bytes
     *   - datetime
     *   - content_type
     *   - dir
     *   - basename
     *   - name
     *   - ext
     *   - alt
     *   - _usage
     * ---
	 *
     * [--axis=<axis>]
     * : 'x' = flip left-to-right; 'y' = flip top-to-bottom.
     * ---
     * default: y
     * options:
     *   - x
     *   - y
	 *
     * [--output_type=<type>]
     * : Output image type.  'auto' = no change.
	 *
     * [--output_dpi=<integer>]
     * : Output image resolution.  '0' = no change.
	 *
     * [<output_file>]
     * : Destination image path, overrides configured cache path and image type.
	 *
     * ## EXAMPLES
     *
     *   wp --allow-root img_process flip --axis=x ./atest.gif
     *   wp --allow-root img_process flip --axis=y ./atest.gif
     *
     * @when after_wp_load
	 */
    public function flip( $args, $assoc_args )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('args','assoc_args'),true), E_USER_NOTICE); }

        $axis   = strtolower($assoc_args['axis'] ?? '');
        if (!in_array($axis,array('x','y')))
        {
            $options = get_option( IMG_PROCESSOR_OPTS, array() );
            $axis = $options['axis'] ?? 'y';
        }

        $transform = array(
            'type'      => 'flip'
            , 'axis'    => $axis
            );
        $image = $this->img_cache( $args, $assoc_args, $transform );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('args','assoc_args','transform','image'),true), E_USER_NOTICE); }
        if ($image) { $this->img_result( $args, $assoc_args, $image ); }
	}


	/**
	 * Rotate the image clockwise.
	 * Background RGBa applied iff degrees % 90 != 0.  GD default is black; Imagick default is transparent.
	 *
     * ## OPTIONS
     *
     * <source>
     * : Source image file path or URL.
	 *
	 * [--output=<format>]
     * : Command result format : info|file|..
     * ---
     * default: file
     * options:
     *   - info
     *   - json
     *   - img
     *   - file
     *   - url
     *   - img
     *   - width
     *   - height
     *   - type
     *   - readable
     *   - writeable
     *   - bytes
     *   - datetime
     *   - content_type
     *   - dir
     *   - basename
     *   - name
     *   - ext
     *   - alt
     *   - _usage
     * ---
	 *
     * --degrees=<integer>
     * : Number of degrees to rotate.
	 *
     * [--color_r=<integer>]
     * : Background color 0-255
	 *
     * [--color_g=<integer>]
     * : Background color 0-255
	 *
     * [--color_b=<integer>]
     * : Background color 0-255
	 *
     * [--color_a=<float>]
     * : Background opacity 0.0-1.0
	 *
     * [--output_type=<type>]
     * : Output image type.  'auto' = no change.
	 *
     * [--output_dpi=<integer>]
     * : Output image resolution.  '0' = no change.
	 *
     * [<output_file>]
     * : Destination image path, overrides configured cache path and image type.
	 *
     * ## EXAMPLES
     *
     *   wp --allow-root img_process rotate --degrees=90 ./atest.gif
     *   wp --allow-root img_process rotate --degrees=-45 ./atest.gif
     *   wp --allow-root img_process rotate --degrees=45 --color_a=1 ./atest.gif
     *   wp --allow-root img_process rotate --degrees=45 --color_r=255 --color_g=255 --color_b=255 ./atest.gif
     *
     * @when after_wp_load
	 */
    public function rotate( $args, $assoc_args )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('args','assoc_args'),true), E_USER_NOTICE); }

        $degrees = (isset($assoc_args['degrees'])) ? intval($assoc_args['degrees']) : 0;
        if (empty($degrees))
        {
            $options = get_option( IMG_PROCESSOR_OPTS, array() );
            $degrees = $options['degrees'] ?? 90;
        }

        $transform = array(
            'type'      => 'rotate'
            , 'degrees' => $degrees
            );
        // color params
        if (isset($assoc_args['color_r'])) { $transform['color_r'] = intval($assoc_args['color_r']); }
        if (isset($assoc_args['color_g'])) { $transform['color_g'] = intval($assoc_args['color_g']); }
        if (isset($assoc_args['color_b'])) { $transform['color_b'] = intval($assoc_args['color_b']); }
        if (isset($assoc_args['color_a'])) { $transform['color_a'] = floatval($assoc_args['color_a']); }

        $image = $this->img_cache( $args, $assoc_args, $transform );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('args','assoc_args','transform','image'),true), E_USER_NOTICE); }
        if ($image) { $this->img_result( $args, $assoc_args, $image ); }
	}


	/**
	 * Resize image proportionally by 'percent'.
	 *
     * ## OPTIONS
     *
     * <source>
     * : Source image file path or URL.
	 *
	 * [--output=<format>]
     * : Command result format : info|file|..
     * ---
     * default: file
     * options:
     *   - info
     *   - json
     *   - img
     *   - file
     *   - url
     *   - img
     *   - width
     *   - height
     *   - type
     *   - readable
     *   - writeable
     *   - bytes
     *   - datetime
     *   - content_type
     *   - dir
     *   - basename
     *   - name
     *   - ext
     *   - alt
     *   - _usage
     * ---
	 *
     * --percent=<integer>
     * : Scale of the result relative to the input.  '100' = no change
	 *
     * [--output_type=<type>]
     * : Output image type.  'auto' = no change.
	 *
     * [--output_dpi=<integer>]
     * : Output image resolution.  '0' = no change.
	 *
     * [<output_file>]
     * : Destination image path, overrides configured cache path and image type.
	 *
     * ## EXAMPLES
     *
     *   wp --allow-root img_process scale --percent=150 ./atest.gif
     *
     * @when after_wp_load
	 */
    public function scale( $args, $assoc_args )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('args','assoc_args'),true), E_USER_NOTICE); }

        $percent = (isset($assoc_args['percent'])) ? intval($assoc_args['percent']) : 100;
        if (empty($percent))
        {
            $options = get_option( IMG_PROCESSOR_OPTS, array() );
            $percent = $options['percent'] ?? 100;
        }

        $transform = array(
            'type'      => 'scale'
            , 'percent' => $percent
            );
        $image = $this->img_cache( $args, $assoc_args, $transform );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('args','assoc_args','transform','image'),true), E_USER_NOTICE); }
        if ($image) { $this->img_result( $args, $assoc_args, $image ); }
	}


	/**
	 * Resize and crop image proportionally by 'percent'.
	 *
     * ## OPTIONS
     *
     * <source>
     * : Source image file path or URL.
	 *
	 * [--output=<format>]
     * : Command result format : info|file|..
     * ---
     * default: file
     * options:
     *   - info
     *   - json
     *   - img
     *   - file
     *   - url
     *   - img
     *   - width
     *   - height
     *   - type
     *   - readable
     *   - writeable
     *   - bytes
     *   - datetime
     *   - content_type
     *   - dir
     *   - basename
     *   - name
     *   - ext
     *   - alt
     *   - _usage
     * ---
	 *
     * --percent=<integer>
     * : Scale of the result relative to the input.  '100' = no change
	 *
     * [--focus=<focus>]
     * : Position of result relative to zoomed image. 'X' = center
     * ---
     * default: X
     * options:
     *   - X
     *   - N
     *   - S
     *   - E
     *   - W
     *   - NE
     *   - NW
     *   - SE
     *   - SW
     * ---
	 *
     * [--width=<integer>]
     * : Width in pixels.
	 *
     * [--height=<integer>]
     * : Height in pixels.
     *
     * [--output_type=<type>]
     * : Output image type.  'auto' = no change.
	 *
     * [--output_dpi=<integer>]
     * : Output image resolution.  '0' = no change.
	 *
     * [<output_file>]
     * : Destination image path, overrides configured cache path and image type.
	 *
     * ## EXAMPLES
     *
     *   wp --allow-root img_process zoom --percent=150 ./atest.gif
     *   wp --allow-root img_process zoom --percent=150 --width=100 --height=100 ./atest.gif
     *   wp --allow-root img_process zoom --percent=150 --width=100 --height=100 --focus=NW ./atest.gif
     *
     * @when after_wp_load
	 */
    public function zoom( $args, $assoc_args )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('args','assoc_args'),true), E_USER_NOTICE); }

        $percent = (isset($assoc_args['percent'])) ? intval($assoc_args['percent']) : 100;
        $focus   = (isset($assoc_args['focus']))   ? strval($assoc_args['focus'])   : null;
        $width   = (isset($assoc_args['width']))   ? intval($assoc_args['width'])   : 0;
        $height  = (isset($assoc_args['height']))  ? intval($assoc_args['height'])  : 0;
        if (empty($percent))
        {
            $options = get_option( IMG_PROCESSOR_OPTS, array() );
            $percent = $options['percent'] ?? 100;
        }

        $transform = array(
            'type'      => 'zoom'
            , 'percent' => $percent
            , 'focus'   => $focus
            , 'width'   => $width
            , 'height'  => $height
            );
        $image = $this->img_cache( $args, $assoc_args, $transform );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('args','assoc_args','transform','image'),true), E_USER_NOTICE); }
        if ($image) { $this->img_result( $args, $assoc_args, $image ); }
	}


    /**
     * Run the requested transform.
     * Source image in $args[0] or $assoc_args['source']
     *
     * @param array $args       : command args
     * @param array $assoc_args : command args
     * @param array $transform  : parameters for getImageCache()
     *
     * @returns array   : cached file info
     */
    protected function img_cache( $args, $assoc_args, $transform )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('args','assoc_args','transform'),true), E_USER_NOTICE); }

        $src = $args[0] ?? ($assoc_args['source'] ?? null);
        if (empty($src))
            { WP_CLI::error(__METHOD__." ERROR: missing input source path "); return; }
        if (empty($transform))
            { WP_CLI::error(__METHOD__." ERROR: missing input transform parameters "); return; }

        $path = null;
        if (filter_var($src, FILTER_VALIDATE_URL)) {
            $path = IMG_Processor_Util::getFileExistsRemote( $src );
        } else {
            $options = get_option( IMG_PROCESSOR_OPTS, array() );
            $search_paths = explode("\n", $options['search_path'] ?? '');
            $search_paths[] = ABSPATH;
            $path = IMG_Processor_Util::getFileExistsLocal( $src, $search_paths );
        }
        if (empty($path))
            { WP_CLI::error(__METHOD__." ERROR: invalid input source path '$src'"); return; }

        $params = array(
            'input' => array(
                'file' => $path
                ),
            'transform' => $transform,
            'output' => array(
                'file'   => $args[1] ?? ($assoc_args['output_file'] ?? null)
                , 'type' => $assoc_args['output_type'] ?? null
                , 'dpi'  => $assoc_args['output_dpi']  ?? null
                ),
            );

        if (IMG_PROCESSOR_IMAGICK)
        {
            global $IMG_MGR;
            if (empty($IMG_MGR)) { $IMG_MGR = new IMG_Processor_Imagick(); }
            $result = $IMG_MGR->getImageCache( $params );
        }
        else
        {
            global $GD_MGR;
            if (empty($GD_MGR)) { $GD_MGR = new IMG_Processor_GD(); }
            $result = $GD_MGR->getImageCache( $params );
        }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params','result'),true), E_USER_NOTICE); }
        return $result;
    }


    /**
     * Output image result per $assoc_args['output']
     *
     * @param array $args       : command args
     * @param array $assoc_args : command args
     * @param array $image      : image info from img_cache()
     *
     * @param arg '--output'    : string, info|json|array_keys(info)
     *
     * @returns string output to console
     */
    protected function img_result( $args, $assoc_args, $image )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('args','assoc_args','image'),true), E_USER_NOTICE); }

        $output = $assoc_args['output'] ?? 'file';
        if (empty($image))
            { WP_CLI::error(__METHOD__." ERROR: missing input image info array "); return; }

        if (in_array($output, array_keys($image)))
        {
            WP_CLI::line( print_r($image[$output],true) );
            return;
        }
        if ($output == 'json')
        {
            unset($image['img'],$image['_usage']);
            WP_CLI::line( json_encode($image) );
            return;
        }
        if ($output == 'info')
        {
            WP_CLI::success( print_r($image,true) );
            return;
        }

        WP_CLI::line( print_r($image,true) );
        WP_CLI::error(__METHOD__." ERROR: unknown result output format '$output'");
    }


} // end class IMG_Processor_CLI
