<?php
/**
 * Image proccessor
 *
 * @package           tenandtwo-wp-plugins
 * @subpackage        tenandtwo-img-processor
 * @author            Ten & Two Systems
 * @copyright         2024 Ten & Two Systems
 */


/**
 * Exstensible for additional "transform" params
 *
 * @see IMG_Processor_GD::getImageCache()
 */
class IMG_Processor_GD
{

    /**
     * @var array returned by gd_info()
     */
    public $gd_info;
    /**
     * @var array see http://php.net/manual/en/function.imagetypes.php
     */
    public $gd_image_types;
    /**
     * @var array allowed types for ImageCreateFromType()
     */
    public $gd_image_input;
    /**
     * @var array allowed types for ImageType()
     */
    public $gd_image_output;

    /**
     * init members
     * cleanCache() per gc_probability
     */
    function __construct()
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }
//if (WP_DEBUG) { echo "<pre>"; debug_print_backtrace(); echo "</pre>\n"; }

        $this->gd_info = null;
        $this->gd_image_input = null;   // GIF | JPEG | PNG | WEBP ...
        $this->gd_image_output = null;  // GIF | JPEG | PNG | WEBP ...

        $this->gd_image_types = array(
            0                   => 'UNKNOWN',
            IMAGETYPE_GIF       => 'GIF',       // Graphics Interchange Format
            IMAGETYPE_JPEG      => 'JPEG',      // Joint Photographic Expert Group
            IMAGETYPE_PNG       => 'PNG',       // Portable Network Graphic
            IMAGETYPE_SWF       => 'SWF',       // Flash
            IMAGETYPE_PSD       => 'PSD',       // Photoshop
            IMAGETYPE_BMP       => 'BMP',       // Bitmap
            IMAGETYPE_TIFF_II   => 'TIFF',      // Tagged Image File Format, intel byte order
            IMAGETYPE_TIFF_MM   => 'TIFF',      // Tagged Image File Format, motorola byte order
            IMAGETYPE_JPC       => 'JPC',       // JPEG 2000
            IMAGETYPE_JP2       => 'JP2',       // JPEG 2000
            IMAGETYPE_JPX       => 'JPX',       // JPEG 2000, extended
            IMAGETYPE_JB2       => 'JB2',       // JBIG2, bi-tonal
            IMAGETYPE_SWC       => 'SWC',       // Flash, compressed
            IMAGETYPE_IFF       => 'IFF',       // Amiga Interchange
            IMAGETYPE_WBMP      => 'WBMP',      // Windows/Wireless Bitmap
            IMAGETYPE_XBM       => 'XBM',       // X Monochrome Bitmap
            IMAGETYPE_ICO       => 'ICO',       // Microsoft Windows Icon
            IMAGETYPE_WEBP      => 'WEBP',      // Google
            IMAGETYPE_AVIF      => 'AVIF',      // AV1 Image File Format
            );
            // TGA (Truevision Graphics Adapter)
            // XPM (X Window PixMap)

        if (ini_get('session.gc_probability') >= rand(1,ini_get('session.gc_divisor')))
            { $cache = $this->cleanCache(); }
    }

    function __destruct()
    {
//if (WP_DEBUG) { trigger_error(__METHOD__, E_USER_NOTICE); }
    }

    /**
     * Check image resource/class
     */
    public function is_image( $image )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('image'),true), E_USER_NOTICE); }
        return ($image instanceof GdImage
            || is_resource( $image ) && 'gd' === get_resource_type( $image ));
    }

    /**
     * Populate this->gd_info = array(
     *
     * <code>
     * this->gd_info = array(
     *     'GD Version'         => {string}
     *     , 'FreeType Support'   => {bool}
     *     , 'FreeType Linkage'   => {string}
     *     , 'GIF Read Support'   => {bool}
     *     , 'GIF Create Support' => {bool}
     *     , 'JPEG Support'       => {bool}
     *     , 'PNG Support'        => {bool}
     *     , 'WBMP Support'       => {bool}
     *     , 'XPM Support'        => {bool}
     *     , 'XBM Support'        => {bool}
     *     , 'WebP Support'       => {bool}
     *     , 'BMP Support'        => {bool}
     *     , 'AVIF Support'       => {bool}
     *     , 'TGA Support'        => {bool}
     *     , 'JIS-mapped Japanese Font Support' => {bool}
     *     );
     * </code>
     *
     * @param none
     * @return bool           true for success ; false for error
     */
    public function initHandler()
    {
//if (WP_DEBUG) { trigger_error(__METHOD__, E_USER_NOTICE); }
        if (is_array($this->gd_info) && count($this->gd_info))
            { return true; }
        if (!IMG_PROCESSOR_GD)
            { trigger_error(__METHOD__." ERROR: GD extension not available. ", E_USER_NOTICE); return false; }

        $this->gd_info = gd_info();
        if (!(is_array($this->gd_info) && count($this->gd_info)))
            { trigger_error(__METHOD__." ERROR: nothing returned by gd_info()", E_USER_NOTICE); return false; }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(array('gd_info' => $this->gd_info),true), E_USER_NOTICE); }

        $this->gd_image_input = array();
        foreach ($this->gd_image_types as $key => $val) {
            eval("\$defined = defined('IMG_".$val."');");
            if (!$defined)
                { continue; }
            eval("\$allowed = (imagetypes() & IMG_".$val.");");
            if ($allowed)
                { $this->gd_image_input[] = $val; }
        }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(array('gd_image_input' => $this->gd_image_input),true), E_USER_NOTICE); }

        $this->gd_image_output = array();
        if (!empty($this->gd_info['AVIF Support'])  && function_exists('imageavif'))
            { $this->gd_image_output[] = 'AVIF'; }
        if (!empty($this->gd_info['BMP Support'])  && function_exists('imagebmp'))
            { $this->gd_image_output[] = 'BMP'; }
        if (!empty($this->gd_info['GIF Create Support']) && function_exists('imagegif'))
            { $this->gd_image_output[] = 'GIF'; }
        if (!empty($this->gd_info['JPEG Support']) && function_exists('imagejpeg'))
            { $this->gd_image_output[] = 'JPEG'; }
        if (!empty($this->gd_info['PNG Support']) && function_exists('imagepng'))
            { $this->gd_image_output[] = 'PNG'; }
        if (!empty($this->gd_info['WBMP Support']) && function_exists('imagewbmp'))
            { $this->gd_image_output[] = 'WBMP'; }
        if (!empty($this->gd_info['WebP Support'])  && function_exists('imagewebp'))
            { $this->gd_image_output[] = 'WEBP'; }
        if (!empty($this->gd_info['XBM Support'])  && function_exists('imagexbm'))
            { $this->gd_image_output[] = 'XBM'; }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(array('gd_image_output' => $this->gd_image_output),true), E_USER_NOTICE); }

        return true;
    }

    /**
     * Get info for existing image file
     * Supported 'file' protocols: https://www.php.net/manual/en/wrappers.php
     *
     * @param array $params     required: file
     * - params['file'] : (string) local absolute filepath or supported remote url
     * - params['alt']  : (string) img alt attribute, default to basename
     *
     * @return mixed  : false for error, or array
     * - file         : (string)  local filepath or remote url
     * - width        : (integer) pixels
     * - height       : (integer) pixels
     * - type         : (string)  from this->gd_image_input
     * - content_type : (string)  for output
     * - readable     : (bool)
     * - writeable    : (bool)
     * - bytes        : (integer) filesize                          (123456)
     * - resolution   : (string)  DPI                               (72,72)
     * - datetime     : (string)  modificaiton date                 (2024-01-01 12:00:00)
     * - dir          : (string)  absolute path to image directory  (/www/.../html/pics/)
     * - basename     : (string)  filename with extension           (mypic.jpg)
     * - name         : (string)  filename without extension        (mypic)
     * - ext          : (string)  filename extension                (.jpg)
     * - url          : (string)  local path or remote url
     * - alt          : (string)  original file name
     * - img          : (string)  default <img/> tag
     */
    public function getImageFileInfo( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        if (!$this->initHandler()) { return false; }

        if (empty($params['file']))
            { trigger_error(__METHOD__." ERROR: missing param 'file' ", E_USER_NOTICE); return false; }

        $exists = false;
        $remote = filter_var($params['file'], FILTER_VALIDATE_URL);
        $headers = null;
        if ($remote) {
            $headers = @get_headers($params['file'], 1);
            $headers = array_change_key_case($headers,CASE_LOWER);
            $exists = (!empty($headers[0]) && strpos($headers[0],'404') === false);
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('headers','exists'),true), E_USER_NOTICE); }
        } else {
            $exists = is_file($params['file']);
        }
        if (!$exists)
            { trigger_error(__METHOD__." ERROR: Invalid path ; file does not exist.  (".$params['file'].") ", E_USER_NOTICE); return false; }

        $bytes = ($remote)
            ? ((strcasecmp($headers[0], 'HTTP/1.1 200 OK') != 0) ? $headers['content-length'][1] : $headers['content-length'])
            : filesize($params['file']);

        $imagesize = getimagesize($params['file']);
        if (!$imagesize)
            { trigger_error(__METHOD__." ERROR: Invalid path ; image not recognized.  (".$params['file'].") ", E_USER_NOTICE); return false; }

        $width        = $imagesize[0];
        $height       = $imagesize[1];
        $type         = $this->gd_image_types[$imagesize[2]] ?: '';
        $content_type = image_type_to_mime_type($imagesize[2]) ?: '';
        $readable     = in_array($type,$this->gd_image_input) && ($remote || is_readable($params['file']));
        $writeable    = !$remote && is_writeable($params['file']);
        //$resolution   = imageresolution( $image );
        $datetime     = date("Y-m-d H:i:s", @filemtime($params['file']) ?: time());

        $pathinfo = pathinfo($params['file']);  // get 'dirname', 'basename', 'extension', 'filename'
        $img_alt = $params['alt'] ?? $pathinfo['basename'];

        if ($remote) {
            $url = $params['file'];
        } elseif (strpos($params['file'],ABSPATH) !== false) {
            $url = strtr($params['file'], array(ABSPATH => '/'));
        }
        $img_tag = ($url) ? '<img src="' . $url . '" ' . $imagesize[3] . ' alt="' . $img_alt . '" />' : '';

        $result = array(
            'img'            => $img_tag
            , 'file'         => $params['file']
            , 'url'          => $url
            , 'width'        => $width
            , 'height'       => $height
            , 'type'         => $type
            , 'content_type' => $content_type
            , 'readable'     => $readable
            , 'writeable'    => $writeable
            , 'bytes'        => $bytes
            //, 'resolution'   => $resolution[0].','.$resolution[1]
            , 'datetime'     => $datetime
            , 'dir'          => $pathinfo['dirname'].'/'
            , 'basename'     => $pathinfo['basename']
            , 'name'         => $pathinfo['filename']
            , 'ext'          => $pathinfo['extension']
            , 'alt'          => $img_alt
            );

//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }
if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Same as getImage(), with results cached at options['cache_path'] or 'dir' param
     *
     * @uses IMG_Processor_GD::getImage()
     *
     * @param array $params
     * - input         : (array) see getImage()
     * - transform     : (array) see getImage()
     * - output        : (array) see getImage()
     * - output['file'] : (string)  override cache dir+name+type for getImage()
     * - output['type'] : (string)  default = input image type
     * - output['dpi']  : (integer) default = input image resolution
     * - output['nocache'] : (bool)
     *
     * @return bool             false for error, or array image file info
     */
    public function getImageCache( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        $options = get_option( IMG_PROCESSOR_OPTS, array() );

        if (empty($params['input']))
            { trigger_error(__METHOD__." ERROR: missing param array 'input' ", E_USER_NOTICE); return false; }
        if (empty($params['transform']))
            { $params['transform'] = array(); }
        if (empty($params['output']))
            { $params['output'] = array(); }

        // get source image info
        $info = $this->getImageFileInfo( $params['input'] );
        if (!$info) { return false; }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('info'),true), E_USER_NOTICE); }
        $info['_usage'] = IMG_Processor_Util::getImageCommands( $params, $info );

        // no transform, no cache ; return source info
        if (empty($params['transform']['type']) || $params['transform']['type'] == 'none')
            { return $info; }

        $output_type = $info['type']; // original vs pref vs param
        if (!empty($options['output_type']) && in_array(strtoupper($options['output_type']),$this->gd_image_output))
            { $output_type = $options['output_type']; }
        if (!empty($params['output']['type']) && in_array(strtoupper($params['output']['type']),$this->gd_image_output))
            { $output_type = $params['output']['type']; }

        $output_dpi = 0;             // original vs pref vs param
        if (!empty($options['output_dpi']))
            { $output_dpi = $options['output_dpi']; }
        if (isset($params['output']['dpi']) && intval($params['output']['dpi']) >= 0)
            { $output_dpi = $params['output']['dpi']; }

        $params['output']['nocache'] = !empty($params['output']['nocache']);

        // default cache location
        $params['output']['dir']  = rtrim($options['cache_path'],'/').'/';
        $params['output']['name'] = null;
        $params['output']['type'] = $output_type;
        $params['output']['dpi']  = intval($output_dpi);
        // override cache location
        if (!empty($params['output']['file']))
        {
            $pathinfo = pathinfo($params['output']['file']);  // get 'dirname', 'basename', 'extension', 'filename'
            $params['output']['dir']  = rtrim($pathinfo['dirname'],'/').'/';
            $params['output']['name'] = $pathinfo['filename'];
            $params['output']['type'] = $pathinfo['extension'];
        }

        // output name
        if (empty($params['output']['nocache']) && empty($params['output']['file']))
        {
            // get output directory + name + type
            $params['output']['dir'] .= IMG_Processor_Util::getHash(array('data' => $info['dir'], 'method' => 'crc32b')) . '/';
            $params['output']['dir'] .= strtr($info['name'].'_'.strtolower($info['type']), array('.' => '_')) . '/';
            //$params['output']['dir'] .= strtr($info['name'].'_'.$info['ext'], array('.' => '_')) . '/';

            // get output filename
            switch($params['transform']['type']) {
                case 'fit':
                case 'fill':
                case 'resize':
                    $params['output']['name'] = $params['transform']['type']
                        . '_' . ($params['transform']['width']  ?? '0')
                        . '_' . ($params['transform']['height'] ?? '0')
                        . '';
                    break;
                case 'crop':
                    $params['output']['name'] = $params['transform']['type']
                        . '_' . ($params['transform']['top']    ?? '0')
                        . '_' . ($params['transform']['left']   ?? '0')
                        . '_' . ($params['transform']['width']  ?? '0')
                        . '_' . ($params['transform']['height'] ?? '0')
                        . '';
                    break;
                case 'trim':
                    $params['output']['name'] = $params['transform']['type']
                        . '_' . ($params['transform']['top']    ?? '0')
                        . '_' . ($params['transform']['left']   ?? '0')
                        . '_' . ($params['transform']['bottom'] ?? '0')
                        . '_' . ($params['transform']['right']  ?? '0')
                        . '';
                    break;
                case 'border':
                    $params['output']['name'] = $params['transform']['type']
                        . '_' . ($params['transform']['top']    ?? '0')
                        . '_' . ($params['transform']['left']   ?? '0')
                        . '_' . ($params['transform']['bottom'] ?? '0')
                        . '_' . ($params['transform']['right']  ?? '0')
                        . '_' . sprintf("%02x%02x%02x", ($params['transform']['color_r'] ?? '0'), ($params['transform']['color_g'] ?? '0'), ($params['transform']['color_b'] ?? '0'))
                        . '';
                    break;
                case 'flip':
                    $params['transform']['axis'] = $params['transform']['axis'] ?? 'y';
                    $params['output']['name'] = $params['transform']['type']
                        . '_' . (($params['transform']['axis'] == 'x') ? 'x' : 'y')
                        . '';
                    break;
                case 'rotate':
                    $params['transform']['degrees'] = $params['transform']['degrees'] % 360 ?? '0';
                    if ($params['transform']['degrees'] < 0)
                        { $params['transform']['degrees'] = 360 + $params['transform']['degrees']; }
                    $params['output']['name'] = $params['transform']['type']
                        . '_' . ($params['transform']['degrees'] ?? '0')
                        . '_' . sprintf("%02x%02x%02x", ($params['transform']['color_r'] ?? '0'), ($params['transform']['color_g'] ?? '0'), ($params['transform']['color_b'] ?? '0'))
                        . '';
                    break;
                case 'scale':
                    $params['output']['name'] = $params['transform']['type']
                        . '_' . ($params['transform']['percent'] ?? '0')
                        . '';
                    break;
                case 'zoom':
                    $params['output']['name'] = $params['transform']['type']
                        . '_' . ($params['transform']['percent'] ?? '0')
                        . '_' . strtolower($params['transform']['focus'] ?? 'x')
                        . '_' . ($params['transform']['width']   ?? '0')
                        . '_' . ($params['transform']['height']  ?? '0')
                        . '';
                    break;

// filters
                case 'grayscale':
                case 'colorize':
                case 'brightness':
                case 'contrast':
                case 'smooth':
                case 'pixelate':
                case 'gaussian_blur':
                case 'selective_blur':
                case 'edgedetect':
                case 'emboss':
                case 'mean_removal':
                case 'negate':
                case 'none':
                default:
                    $params['output']['name'] = $info['name'];
                    $params['output']['nocache'] = true;
                    break;
            }
            if (WP_DEBUG) { $params['output']['name'] = 'gd_'.$params['output']['name']; }
        }

        // check for cached result
        if (empty($params['output']['nocache']))
        {
            // setup/check output cache directory
            $cache_file = $params['output']['dir'] . strtolower($params['output']['name'].'.'.$params['output']['type']);
            $cache_exists = false;

            clearstatcache();
            if (!is_dir($params['output']['dir'])) {
                umask(0000);
                if (!mkdir($params['output']['dir'], 0775, true))
                    { trigger_error(__METHOD__." ERROR: could not create folder '".$params['output']['dir']."' ", E_USER_NOTICE); return false; }
            } else {
                $cache_exists = is_file($cache_file);
            }
if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('cache_file','cache_exists'),true), E_USER_NOTICE); }

            // delete cache file if source updated
            if ($cache_exists && filemtime($cache_file) < @filemtime($info['file'])) {
                $cache_exists = false;
                if (!unlink($cache_file))
                    { trigger_error(__METHOD__." ERROR: could not remove out-dated file '".$cache_file."' ", E_USER_NOTICE); return false; }
            }

            // return existing cache info
            if ($cache_exists) {
                $subparams = array(
                    'file'  => realpath($cache_file),
                    'alt'   => $info['basename'],  // original filename
                    );
                $result = $this->getImageFileInfo( $subparams );
                if (!$result) { return false; }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('cache_exists','params','result'),true), E_USER_NOTICE); }

$result['_usage'] = IMG_Processor_Util::getImageCommands( $params, $result );
                return $result;
            }
        }

        // if nocache, clear output params ; getImage() will return 'data' not 'file'
        if (!empty($params['output']['nocache'])) { $params['output'] = array(); }

        // return new cache info
        $result = $this->getImage( $params );
        if (!$result) { return false; }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('cache_exists','params','result'),true), E_USER_NOTICE); }

$result['_usage'] = IMG_Processor_Util::getImageCommands( $params, $result );
        return $result;
    }

    /**
     * Delete all or unused items in image cache directory
     * remove all || files not created|modified|accessed for expire_value
     *
     * @param array $params
     * - cache_path     : (string) directory path
     * - expire_type    : (string) atime|amin|ctime|cmin|mtime|mmin = [access|create|modify][days|minutes]
     * - expire_value   : (integer) number of expire_units
     *
     * @return mixed    : false for error, or array for cache directory
     */
    public function cleanCache( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        //$valid_expire_type = array('all','Bmin','Btime', 'amin','atime', 'cmin','ctime', 'mmin','mtime',);
        $valid_expire_type = array('all','atime','amin','ctime','cmin','mtime','mmin');

        $options = get_option( IMG_PROCESSOR_OPTS, array() );
        $cache_path   = $params['cache_path']   ?? $options['cache_path'];
        $expire_type  = $params['expire_type']  ?? $options['expire_type'];
        $expire_value = $params['expire_value'] ?? $options['expire_value'];

        if (!is_dir($cache_path))
            { trigger_error(__METHOD__." ERROR: cache directory not found '".$cache_path."' ", E_USER_NOTICE); return false; }
        if (!is_writeable($cache_path))
            { trigger_error(__METHOD__." ERROR: cache directory not writeable '".$cache_path."' ", E_USER_NOTICE); return false; }

        if (!in_array($expire_type,$valid_expire_type))
            { trigger_error(__METHOD__." ERROR: invalid cache expire_type '".$expire_type."' ", E_USER_NOTICE); return false; }
        if (!(0 <= intval($expire_value)))
            { trigger_error(__METHOD__." ERROR: invalid cache expire_value '".$expire_value."' ", E_USER_NOTICE); return false; }

        $cmd_files = "find ".$cache_path."/. -type f -name '*.*'"
                . (($expire_type == "all") ? "" : " -".$expire_type." +".$expire_value)
                . " -delete";
        $cmd_dirs = "find ".$cache_path."/. -depth -type d -empty -delete";

        // run command-line
        @exec($cmd_files . ' ; ' . $cmd_dirs);

        $result = array(
            'cache_path'     => $cache_path
            , 'expire_type'  => $expire_type
            , 'expire_value' => $expire_value
            );

if (WP_DEBUG) { trigger_error(__METHOD__." : exec($cmd_files ; $cmd_dirs) ", E_USER_NOTICE); }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params','result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Read source image, transform, output result
     *
     * @uses IMG_Processor_GD::initHandler()
     * @uses IMG_Processor_GD::readImage()
     * @uses IMG_Processor_GD::transformImage()
     * @uses IMG_Processor_GD::writeImage()
     *
     * @param array $params
     * REQUIRED : one of the following
     * - file : (string) absolute filepath or supported remote url
     * - data : (string) dbresult || result of previous getImage() call
     *
     * transform              : (array)
     * - transform['type']    : (string) none || fit || fill || resize || crop || trim || border
     *                          || flip || rotate || scale || zoom
     *
     * if type == fit, one param required
     * - transform['width']   : (integer) pixels
     * - transform['height']  : (integer) pixels
     *
     * if type == resize or type == fill, both params required
     * - transform['width']   : (integer) pixels
     * - transform['height']  : (integer) pixels
     *
     * if type == crop or type == border, one param required
     * - transform['top']     : (integer) pixels
     * - transform['left']    : (integer) pixels
     * - transform['bottom']  : (integer) pixels
     * - transform['right']   : (integer) pixels
     *
     * if type == border, optional
     * - transform['color_r'] : (integer) bgcolor 0..255
     * - transform['color_g'] : (integer) bgcolor 0..255
     * - transform['color_b'] : (integer) bgcolor 0..255
     *
     * if type == flip, param required
     * - transform['axis'] : (string) x || y
     *
     * if type == rotate, param 'degrees' required
     * - transform['degrees'] : (integer) eg, 90 || 180 || 270
     * - transform['color_r'] : (integer) bgcolor 0..255
     * - transform['color_g'] : (integer) bgcolor 0..255
     * - transform['color_b'] : (integer) bgcolor 0..255
     *
     * if type == scale, param required
     * - transform['percent'] : (string) eg 50% || 200
     *
     * if type == zoom, percent param required
     * - transform['percent'] : (string) eg 50% || 200
     * - transform['focus']   : (string) X|N|S|E|W|NE|NW|SE|SW
     * - transform['width']   : (integer) pixels
     * - transform['height']  : (integer) pixels
     *
     *
     * output               : (array)
     * - default : output to input dir + name + type
     * - returns raw data if !(force || replace || file || dir || name)
     *
     * output['force']      : (bool)
     * - true = overwrite existing file if necessary
     *
     * output['replace']    : (bool)
     * - true = delete original if writing new file
     *
     * output['dir']        : (string) absolute path to cache directory
     * - default = input file's directory path
     *
     * output['name']       : (string) cache file's basename
     * - default = input file's basename
     *
     * output['type']       : (string) from gd_image_output
     * - default = input type || PNG if string input
     * - determines output .ext
     *
     * output['dpi']        : (integer) resolution
     * - default = input resolution
     *
     * @return bool             false for error, or image info array
     */
    public function getImage( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        if (!$this->initHandler()) { return false; }

        $err = false;
        while (true)
        {
//if (WP_DEBUG) { trigger_error(__METHOD__." : readImage() ", E_USER_NOTICE); }
            $params['input_result']     = $this->readImage( $params );
            if (!$params['input_result'])     { $err = true; break; }
//if (WP_DEBUG) { trigger_error(__METHOD__." : transformImage() ", E_USER_NOTICE); }
            $params['transform_result'] = $this->transformImage( $params );
            if (!$params['transform_result']) { $err = true; break; }
//if (WP_DEBUG) { trigger_error(__METHOD__." : writeImage() ", E_USER_NOTICE); }
            $params['output_result']    = $this->writeImage( $params );
            if (!$params['output_result'])    { $err = true; break; }
            break;
        }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        // free image resources
        if (isset($params['input_result']['image']) && $this->is_image($params['input_result']['image'])) {
            imagedestroy($params['input_result']['image']);
            $params['input_result']['image'] = null;
        }
        if (isset($params['transform_result']['image']) && $this->is_image($params['transform_result']['image'])) {
            imagedestroy($params['transform_result']['image']);
            $params['transform_result']['image'] = null;
        }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        // return false for error
        if ($err) { return false; }

        $result = $params['output_result'];
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Create an image object from existing file or data string
     *
     * @uses IMG_Processor_GD::read_image_data()
     * @uses IMG_Processor_GD::read_image_file()
     *
     * @param array $params
     * - input['file'] : (string) filepath or supported remote url
     * - input['data'] : (string) dbresult || result of previous getImage() call
     *
     * @return bool             false for error, or image data array
     */
    protected function readImage( $params )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }
        if (empty($params['input']))
            { trigger_error(__METHOD__." ERROR: missing param array 'input' ", E_USER_NOTICE); return false; }

        if (empty($params['input']['file'])) { $params['input']['file'] = null; }
        if (empty($params['input']['data'])) { $params['input']['data'] = null; }
        if (empty($params['input']['file']) && empty($params['input']['data']))
            { trigger_error(__METHOD__." ERROR: missing param 'file' || 'data' ", E_USER_NOTICE); return false; }

        $result = (!empty($params['input']['data']))
                ? $this->read_image_data( $params )
                : $this->read_image_file( $params );
        if (!$result) { return false; }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Dispatch according to params['transform']['type']
     * @see http://www.phpclasses.org/browse/file/25856.html for transform examples
     *
     * @uses IMG_Processor_GD::transform_fit()
     * @uses IMG_Processor_GD::transform_fill()
     * @uses IMG_Processor_GD::transform_resize()
     * @uses IMG_Processor_GD::transform_crop()
     * @uses IMG_Processor_GD::transform_trim()
     * @uses IMG_Processor_GD::transform_border()
     * @uses IMG_Processor_GD::transform_flip()
     * @uses IMG_Processor_GD::transform_rotate()
     * @uses IMG_Processor_GD::transform_scale()
     * @uses IMG_Processor_GD::transform_zoom()
     *
     * @param array $params     transform ; output ; input_result
     * - transform['type'] : (string) default == none
     *
     * @return bool             false for error, or transformed image data
     */
    protected function transformImage( $params )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        $result = null;

        if (empty($params['transform']['type']))
            { $params['transform']['type'] = 'none'; }

        switch($params['transform']['type']) {
            case 'none':
                break;

            case 'fit':
                $result = $this->transform_fit( $params );
                break;
            case 'fill':
                $result = $this->transform_fill( $params );
                break;
            case 'resize':
                $result = $this->transform_resize( $params );
                break;
            case 'crop':
                $result = $this->transform_crop( $params );
                break;
            case 'trim':
                $result = $this->transform_trim( $params );
                break;
            case 'border':
                $result = $this->transform_border( $params );
                break;
            case 'flip':
                $result = $this->transform_flip( $params );
                break;
            case 'rotate':
                $result = $this->transform_rotate( $params );
                break;
            case 'scale':
                $result = $this->transform_scale( $params );
                break;
            case 'zoom':
                $result = $this->transform_zoom( $params );
                break;
// filters
            case 'grayscale':

            default:
                trigger_error(__METHOD__." ERROR: unsupported transform type '".$params['transform']['type']."' ", E_USER_NOTICE);
                break;
        }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Dispatch to write_image_data() || write_image_file()
     *
     * @uses IMG_Processor_GD::write_image_data()
     * @uses IMG_Processor_GD::write_image_file()
     *
     * @param array $params
     * - REQUIRED : one of the following
     * - transform_result   : (array) image object and info
     * - input_result       : (array) image object and info
     * - OPTIONAL
     * - output['force']   : (bool) overwrite existing file if necessary
     * - output['replace'] : (bool) delete original if writing to new file
     * - output['file']    : (string) absolute path to new image file
     * - output['dir']     : (string) path to image file's parent directory
     * - output['name']    : (string) image file's basename
     * - output['type']    : (string) output image type from gd_image_output
     *
     * @return bool             false for error, or image info array
     */
    protected function writeImage( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        // output file specs
        if (!isset($params['output']['force']))   { $params['output']['force'] = null; }
        if (!isset($params['output']['replace'])) { $params['output']['replace'] = null; }
        if (!isset($params['output']['file']))    { $params['output']['file'] = null; }
        if (!isset($params['output']['dir']))     { $params['output']['dir'] = null; }
        if (!isset($params['output']['name']))    { $params['output']['name'] = null; }
        if (!isset($params['output']['type']))    { $params['output']['type'] = null; }
        if (!isset($params['output']['dpi']))     { $params['output']['dpi'] = null; }

        if (empty($params['output']['type']) && !empty($params['input_result']['type']))
            { $params['output']['type'] = $params['input_result']['type']; }

        $result = (!($params['output']['force']
                || $params['output']['replace']
                || $params['output']['file']
                || $params['output']['dir']
                || $params['output']['name']
                ))
            ? $this->write_image_data( $params )
            : $this->write_image_file( $params );

//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Create an image object from data string
     *
     * @param array $params
     * - input['data'] : (string) required
     *
     * @return bool             false for error, or image info array
     * - image  : (object) see php::ImageCreateFromString()
     * - width  : (integer)
     * - height : (integer)
     * - type   : (string)
     */
    protected function read_image_data( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        if (!(isset($params['input']['data']) && strlen($params['input']['data'])))
            { trigger_error(__METHOD__." ERROR: missing param 'data' ", E_USER_NOTICE); return false; }

        $image = @imagecreatefromstring( $params['input']['data'] );

        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: failed to create image resource for data", E_USER_NOTICE); return false; }

        $result = array(
            'image'     => $image
            , 'width'   => imagesx($image)
            , 'height'  => imagesy($image)
            , 'type'    => 'PNG'    // not really
            );

//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Create an image object from existing file
     *
     * @uses IMG_Processor_GD::getImageFileInfo()
     *
     * @param array $params
     * - input : (array)
     *
     * @return mixed             false for error, or array
     * - image : (object) see php::ImageCreateFromXYZ()
     */
    protected function read_image_file( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        $info = $this->getImageFileInfo( $params['input'] );
        if (!$info) { return false; }

        if (!strlen($info['type']))
            { trigger_error(__METHOD__." ERROR: missing param 'type' ", E_USER_NOTICE); return false; }
        if (!strlen($info['file']))
            { trigger_error(__METHOD__." ERROR: missing param 'file' ", E_USER_NOTICE); return false; }
        if (!$info['readable'])
            { trigger_error(__METHOD__." ERROR: unreadable input type '".$info['type']."' ", E_USER_NOTICE); return false; }

//$memory_usage = memory_get_usage(true);
//$image_bytes = $info['bytes'];

//$memory_limit = ini_get('memory_limit');
//$tmp_memory_limit = -1;                       // unlimited
//$tmp_memory_limit = $memory_usage + ($image_bytes * ???);  // current + (image_bytes * N)

//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('memory_usage','image_bytes'),true), E_USER_NOTICE); }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('memory_limit','tmp_memory_limit'),true), E_USER_NOTICE); }

//ini_set('memory_limit',strval($tmp_memory_limit));
        $image = null;
        switch ($info['type']) {
            case 'AVIF':
                $image = @imagecreatefromavif($info['file']);
                break;
            case 'BMP':
                $image = @imagecreatefrombmp($info['file']);
                break;
            case 'GIF':
                $image = @imagecreatefromgif($info['file']);
                break;
            case 'JPEG':
                $image = @imagecreatefromjpeg($info['file']);
                break;
            case 'PNG':
                $image = @imagecreatefrompng($info['file']);
                break;
            case 'WBMP':
                $image = @imagecreatefromwbmp($info['file']);
                break;
            case 'WEBP':
                $image = @imagecreatefromwebp($info['file']);
                break;
            case 'XBM':
                $image = @imagecreatefromxbm($info['file']);
                break;

//             case 'TGA':
//                 $image = @imagecreatefromtga($info['file']);
//                 break;
//             case 'XPM':
//                 $image = @imagecreatefromxpm($info['file']);
//                 break;

            default:
                break;
        }
//ini_set('memory_limit',strval($memory_limit));

//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('image'),true), E_USER_NOTICE); }
        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: failed to create image resource for '".$info['type']."' at '".$info['file']."' ", E_USER_NOTICE); return false; }

        $result = array_merge( array('image' => $image), $info );

//$memory_usage = memory_get_usage(true);
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('memory_usage','result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Returns raw image data in result array
     *
     * @param array $params
     * - REQUIRED
     * - output['type'] : (string) output image type from gd_image_output
     * - REQUIRED - one of the following
     * - transform_result   : (array) image object and info
     * - input_result       : (array) image object and info
     *
     * @return mixed            false for error, or image info array
     * - type         : (string) output image_type
     * - width        : (string) output dimension
     * - height       : (string) output dimension
     * - content_type : (string) eg, image/png
     * - data         : (string) image data
     */
    protected function write_image_data( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        // output image
        $image = null;
        if (isset($params['transform_result']['image']) && $this->is_image($params['transform_result']['image']))
            { $image = $params['transform_result']['image']; }
        else if (isset($params['input_result']['image']) && $this->is_image($params['input_result']['image']))
            { $image = $params['input_result']['image']; }
        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: missing param 'transform_result[image]' || 'input_result[image]' ", E_USER_NOTICE); return false; }

        // output image type
        if (empty($params['output']['type']))
            { trigger_error(__METHOD__." ERROR: missing param 'output[type]' ", E_USER_NOTICE); return false; }
        if (!in_array(strtoupper($params['output']['type']),$this->gd_image_output))
            { trigger_error(__METHOD__." ERROR: unknown output image type '".$params['output']['type']."' ", E_USER_NOTICE); return false; }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

// resolution?

        // write data
        ob_start();                   // start output buffering
        switch (strtoupper($params['output']['type'])) {
            case 'AVIF':
                $content_type = image_type_to_mime_type(IMAGETYPE_AVIF);
                if (!imageavif($image))
                    { trigger_error(__METHOD__." ERROR: imageavif() failed ", E_USER_NOTICE); return false; }
                break;
            case 'BMP':
                $content_type = image_type_to_mime_type(IMAGETYPE_BMP);
                if (!imagebmp($image))
                    { trigger_error(__METHOD__." ERROR: imagebmp() failed ", E_USER_NOTICE); return false; }
                break;
            case 'GIF':
                $content_type = image_type_to_mime_type(IMAGETYPE_GIF);
                if (!imagegif($image))
                    { trigger_error(__METHOD__." ERROR: imagegif() failed ", E_USER_NOTICE); return false; }
                break;
            case 'JPEG':
                $content_type = image_type_to_mime_type(IMAGETYPE_JPEG);
                $quality = 100;
                if (!imagejpeg($image, null, $quality))
                    { trigger_error(__METHOD__." ERROR: imagejpeg() failed ", E_USER_NOTICE); return false; }
                break;
            case 'PNG':
                $content_type = image_type_to_mime_type(IMAGETYPE_PNG);
                if (!imagepng($image))
                    { trigger_error(__METHOD__." ERROR: imagepng() failed ", E_USER_NOTICE); return false; }
                break;
            case 'WBMP':
                $content_type = image_type_to_mime_type(IMAGETYPE_WBMP);
                $foreground = imagecolorallocate($image, 255, 255, 255);
                if (!imagewbmp($image, null, $foreground))
                    { trigger_error(__METHOD__." ERROR: imagewbmp() failed ", E_USER_NOTICE); return false; }
                break;
            case 'WEBP':
                $content_type = image_type_to_mime_type(IMAGETYPE_WEBP);
                $quality = 100;
                if (!imagewebp($image, null, $quality))
                    { trigger_error(__METHOD__." ERROR: imagewebp() failed ", E_USER_NOTICE); return false; }
                break;
            case 'XBM':
                $content_type = image_type_to_mime_type(IMAGETYPE_XBM);
                $foreground = imagecolorallocate($image, 255, 255, 255);
                if (!imagexbm($image, null, $foreground))
                    { trigger_error(__METHOD__." ERROR: imagexbm() failed ", E_USER_NOTICE); return false; }
                break;
            default:
                $content_type = 'text/html';
                trigger_error(__METHOD__." ERROR: unknown output type '".$params['output']['type']."' ", E_USER_NOTICE);
                return false;
        }
        $data = ob_get_contents();    // get buffer content
        ob_end_clean();               // stop output buffering

        $result = array(
            'data'           => $data
            , 'width'        => imagesx($image)
            , 'height'       => imagesy($image)
            , 'type'         => $params['output']['type']
            , 'bytes'        => strlen($data)
            , 'content_type' => $content_type
            );

//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Write image data to disk
     *
     * @uses IMG_Processor_GD::getImageFileInfo()
     *
     * @param array $params
     * - REQUIRED - one of the following
     * - transform_result   : (array) image object and info
     * - input_result       : (array) image object and info
     * - OPTIONAL
     * - output['force']   : (bool) overwrite existing file if necessary
     * - output['replace'] : (bool) delete original if writing to new file
     * - output['file']    : (string) absolute path to cache file
     * - output['dir']     : (string) absolute path to cache directory
     * - output['name']    : (string) cache file's basename
     * - output['type']    : (string) output image type from imagick_types
     * - output['dpi']     : (integer) output image resolution
     *
     * @return bool             false for error, or image info array
     */
    protected function write_image_file( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        // output image
        $image = null;
        if (isset($params['transform_result']['image']) && $this->is_image($params['transform_result']['image']))
            { $image = $params['transform_result']['image']; }
        else if (isset($params['input_result']['image']) && $this->is_image($params['input_result']['image']))
            { $image = $params['input_result']['image']; }
        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: missing param 'transform_result[image]' || 'input_result[image]' ", E_USER_NOTICE); return false; }

        $params['output']['force']   = !empty($params['output']['force']);
        $params['output']['replace'] = !empty($params['output']['replace']);

        // output file specs : input file loc used for default
        $params['output']['file'] = trim($params['input_result']['file']);
        $info = pathinfo($params['output']['file']);  // get 'dirname', 'basename', 'extension', 'filename'
        // output file specs : overrides
        if (empty($params['output']['dir']))
            { $params['output']['dir'] = $info['dirname'].'/'; }
        if (empty($params['output']['name']))
            { $params['output']['name'] = $info['filename']; }
        if (empty($params['output']['type']) || !in_array(strtoupper($params['output']['type']),$this->gd_image_output))
            { $params['output']['type'] = $info['extension']; }

        // build output filename
        $params['output']['dir']  = rtrim($params['output']['dir'],'/').'/';    // trailing slash
        $params['output']['name'] = strtr($params['output']['name']," ","_");   // no spaces
        $params['output']['type'] = strtolower($params['output']['type']);      // lowercase

        $params['output']['file'] = $params['output']['dir']
                . $params['output']['name']
                . '.'
                . (($params['output']['type'] == 'WBMP') ? 'bmp' : $params['output']['type']);
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        // validate destination path
        if (!is_dir($params['output']['dir']))
            { trigger_error(__METHOD__." ERROR: destination directory '".$params['output']['dir']."' not found ", E_USER_NOTICE); return false; }
        if (!is_writable($params['output']['dir']))
            { trigger_error(__METHOD__." ERROR: destination directory '".$params['output']['dir']."' not writable ", E_USER_NOTICE); return false; }

        // check exists/overwrite
        if (empty($params['output']['force']) && is_file($params['output']['file']))
            { trigger_error(__METHOD__." ERROR: file '".$params['output']['file']."' already exists ", E_USER_NOTICE); return false; }

        // set resolution
        if (!empty($params['output']['dpi']))
        {
            $dpi = $params['output']['dpi'];
            $resolution = imageresolution( $image );
            if ($resolution[0] > $dpi || $resolution[1] > $dpi) {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('dpi','resolution'),true), E_USER_NOTICE); }
                imageresolution( $image, $dpi );
            }
        }

        // write file
        switch (strtoupper($params['output']['type'])) {
            case 'AVIF':
                $quality = 82;
                if (!imageavif($image, $params['output']['file'], $quality))
                    { trigger_error(__METHOD__." ERROR: imageavif('".$params['output']['file']."') failed ", E_USER_NOTICE); return false; }
                break;
            case 'BMP':
                if (!imagebmp($image, $params['output']['file']))
                    { trigger_error(__METHOD__." ERROR: imagebmp('".$params['output']['file']."') failed ", E_USER_NOTICE); return false; }
                break;
            case 'GIF':
                if (!imagegif($image, $params['output']['file']))
                    { trigger_error(__METHOD__." ERROR: imagegif('".$params['output']['file']."') failed ", E_USER_NOTICE); return false; }
                break;
            case 'JPEG':
                $quality = 82;
                if (!imagejpeg($image, $params['output']['file'], $quality))
                    { trigger_error(__METHOD__." ERROR: imagejpeg('".$params['output']['file']."') failed ", E_USER_NOTICE); return false; }
                break;
            case 'PNG':
                if (!imagepng($image, $params['output']['file']))
                    { trigger_error(__METHOD__." ERROR: imagepng('".$params['output']['file']."') failed ", E_USER_NOTICE); return false; }
                break;
            case 'WBMP':
                $foreground = imagecolorallocate($image, 255, 255, 255);
                if (!imagewbmp($image, $params['output']['file'], $foreground))
                    { trigger_error(__METHOD__." ERROR: imagewbmp('".$params['output']['file']."') failed ", E_USER_NOTICE); return false; }
                break;
            case 'WEBP':
                $quality = 86;
                if (!imagewebp($image, $params['output']['file'], $quality))
                    { trigger_error(__METHOD__." ERROR: imagewebp('".$params['output']['file']."') failed ", E_USER_NOTICE); return false; }
                break;
            case 'XBM':
                $foreground = imagecolorallocate($image, 255, 255, 255);
                if (!imagexbm($image, $params['output']['file'], $foreground))
                    { trigger_error(__METHOD__." ERROR: imagexbm('".$params['output']['file']."') failed ", E_USER_NOTICE); return false; }
                break;
            default:
                trigger_error(__METHOD__." ERROR: unknown output type '".$params['output']['type']."' ", E_USER_NOTICE);
                return false;
        }

        // return info for new image file
        $subparams = array(
            'file'  => $params['output']['file'],
            'alt'   => $params['input_result']['basename'],  // original filename
            );
        $result = $this->getImageFileInfo( $subparams );
        if (!$result) { return false; }

        // remove old image file, if necessary
        if ($params['output']['replace'] && $params['input_result']['file'] != $params['output']['file']) {
            if (!unlink($params['input_result']['file'])) {
                trigger_error(__METHOD__." ERROR: Could not remove old file '" . $params['input_result']['file'] . "'", E_USER_NOTICE);
                return false;
            }
        }

//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }


    /**
     * Resize image proportionally to fit requested width and/or height
     *
     * @param array $params
     * - REQUIRED
     * - input_result['image']  : (object) input image
     * - REQUIRED - one of the following
     * - transform['width']     : (integer) pixels
     * - transform['height']    : (integer) pixels
     *
     * @return mixed    : false for error, or image info array
     * - image  : (object)
     * - width  : (integer)
     * - height : (integer)
     */
    protected function transform_fit( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        if (!$this->is_image($params['input_result']['image']))
            { trigger_error(__METHOD__." ERROR: missing input image resource ", E_USER_NOTICE); return false; }

        if (empty($params['transform']['width']) && empty($params['transform']['height']))
            { trigger_error(__METHOD__." ERROR: missing transform param 'width' and/or 'height' ", E_USER_NOTICE); return false; }

        // source rectangle default
        $src = array(
            'x'   => 0
            , 'y' => 0
            , 'w' => $params['input_result']['width']
            , 'h' => $params['input_result']['height']
            );
        // destination rectangle default
        $dest = array(
            'x'   => 0
            , 'y' => 0
            , 'w' => intval($params['transform']['width']  ?? 0)
            , 'h' => intval($params['transform']['height'] ?? 0)
            );

        // set requested destination rectangle
        if (!empty($dest['w']) && !empty($dest['h'])) {
            $dest_height = floor(($src['h'] / $src['w']) * $dest['w']);
            $dest_width  = floor(($src['w'] / $src['h']) * $dest['h']);

            if ($dest_height > $dest['h']) {
                $dest['w'] = $dest_width;
            } else {
                $dest['h'] = $dest_height;
            }

        } else if (!empty($dest['w'])) {
            $dest['h'] = floor(($src['h'] / $src['w']) * $dest['w']);

        } else if (!empty($dest['h'])) {
            $dest['w'] = floor(($src['w'] / $src['h']) * $dest['h']);

        }
//$fit_params = $params['transform'];
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('fit_params','src','dest'),true), E_USER_NOTICE); }

        // get output image resource
        $image = @imagecreatetruecolor($dest['w'],$dest['h']);
        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: failed to create destination image resource ", E_USER_NOTICE); return false; }

        // allow transparency in destination image
        imagealphablending($image,true);

        // copy input -> output
        if (!imagecopyresampled(
                $image                              // object, destination image
                , $params['input_result']['image']  // object, source image
                , $dest['x']                        // int, destination x
                , $dest['y']                        // int, destination y
                , $src['x']                         // int, source x
                , $src['y']                         // int, source y
                , $dest['w']                        // int, destination width
                , $dest['h']                        // int, destination height
                , $src['w']                         // int, source width
                , $src['h']                         // int, source height
                ))
            { trigger_error(__METHOD__." ERROR: imagecopyresampled() failed ", E_USER_NOTICE); return false; }

        // final check and return
        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: failed to create destination image resource ", E_USER_NOTICE); return false; }

        $result = array(
            'image'    => $image
            , 'width'  => imagesx($image)
            , 'height' => imagesy($image)
            );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Resize image proportionally to fill requested width and height
     *
     * @param array $params
     * - REQUIRED
     * - input_result['image']  : (object) input image
     * - REQUIRED - one of the following
     * - transform['width']     : (integer) pixels
     * - transform['height']    : (integer) pixels
     *
     * @return mixed    : false for error, or image info array
     * - image  : (object)
     * - width  : (integer)
     * - height : (integer)
     */
    protected function transform_fill( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        if (!$this->is_image($params['input_result']['image']))
            { trigger_error(__METHOD__." ERROR: missing input image resource ", E_USER_NOTICE); return false; }

        if (empty($params['transform']['width']) && empty($params['transform']['height']))
            { trigger_error(__METHOD__." ERROR: missing transform param 'width' and/or 'height' ", E_USER_NOTICE); return false; }

        // source rectangle default
        $src = array(
            'x'   => 0
            , 'y' => 0
            , 'w' => $params['input_result']['width']
            , 'h' => $params['input_result']['height']
            );
        // destination rectangle default
        $dest = array(
            'x'   => 0
            , 'y' => 0
            , 'w' => intval($params['transform']['width']  ?? 0)
            , 'h' => intval($params['transform']['height'] ?? 0)
            );

        // set requested destination rectangle
        if (!empty($dest['w']) && !empty($dest['h'])) {
            $dest_height = floor(($src['h'] / $src['w']) * $dest['w']);
          //$dest_width  = floor(($src['w'] / $src['h']) * $dest['h']);

            if ($dest_height > $dest['h']) {
                $src_height = floor(($dest['h'] / $dest['w']) * $src['w']);
                $src['y'] = ($src['h'] - $src_height)/2;  // center
                $src['h'] = $src_height;
            } else {
                $src_width = floor(($dest['w'] / $dest['h']) * $src['h']);
                $src['x'] = ($src['w'] - $src_width)/2;  // center
                $src['w'] = $src_width;
            }

        } else if (!empty($dest['w'])) {
            $dest['h'] = floor(($src['h'] / $src['w']) * $dest['w']);
            $src['h']  = floor(($dest['h'] / $dest['w']) * $src['w']);

        } else if (!empty($dest['h'])) {
            $dest['w'] = floor(($src['w'] / $src['h']) * $dest['h']);
            $src['w']  = floor(($dest['w'] / $dest['h']) * $src['h']);

        }
//$fill_params = $params['transform'];
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('fill_params','src','dest'),true), E_USER_NOTICE); }

        // get output image resource
        $image = @imagecreatetruecolor($dest['w'],$dest['h']);
        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: failed to create destination image resource ", E_USER_NOTICE); return false; }

        // allow transparency in destination image
        imagealphablending($image,true);

        // copy input -> output
        if (!imagecopyresampled(
                $image                              // object, destination image
                , $params['input_result']['image']  // object, source image
                , $dest['x']                        // int, destination x
                , $dest['y']                        // int, destination y
                , $src['x']                         // int, source x
                , $src['y']                         // int, source y
                , $dest['w']                        // int, destination width
                , $dest['h']                        // int, destination height
                , $src['w']                         // int, source width
                , $src['h']                         // int, source height
                ))
            { trigger_error(__METHOD__." ERROR: imagecopyresampled() failed ", E_USER_NOTICE); return false; }

        // final check and return
        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: failed to create destination image resource ", E_USER_NOTICE); return false; }

        $result = array(
            'image'    => $image
            , 'width'  => imagesx($image)
            , 'height' => imagesy($image)
            );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Stretch image (non-proportional) to fill requested width and height
     *
     * @param array $params
     * - REQUIRED
     * - input_result['image']  : (object) input image
     * - REQUIRED - one of the following
     * - transform['width']     : (integer) pixels
     * - transform['height']    : (integer) pixels
     *
     * @return mixed    : false for error, or image info array
     * - image  : (object)
     * - width  : (integer)
     * - height : (integer)
     */
    protected function transform_resize( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        if (!$this->is_image($params['input_result']['image']))
            { trigger_error(__METHOD__." ERROR: missing input image resource ", E_USER_NOTICE); return false; }

        if (empty($params['transform']['width']) && empty($params['transform']['height']))
            { trigger_error(__METHOD__." ERROR: missing transform param 'width' and/or 'height' ", E_USER_NOTICE); return false; }

        // source rectangle default
        $src = array(
            'x'   => 0
            , 'y' => 0
            , 'w' => $params['input_result']['width']
            , 'h' => $params['input_result']['height']
            );
        // destination rectangle default
        $dest = array(
            'x'   => 0
            , 'y' => 0
            , 'w' => intval($params['transform']['width']  ?? 0)
            , 'h' => intval($params['transform']['height'] ?? 0)
            );

        // set requested destination rectangle
        if (empty($dest['h'])) {
            $dest['h'] = floor(($src['h'] / $src['w']) * $dest['w']);
        }
        if (empty($dest['w'])) {
            $dest['w'] = floor(($src['w'] / $src['h']) * $dest['h']);
        }

//$resize_params = $params['transform'];
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('resize_params','src','dest'),true), E_USER_NOTICE); }

        // get output image resource
        $image = @imagecreatetruecolor($dest['w'],$dest['h']);
        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: failed to create destination image resource ", E_USER_NOTICE); return false; }

        // allow transparency in destination image
        imagealphablending($image,true);

        // copy input -> output
        if (!imagecopyresampled(
                $image                              // object, destination image
                , $params['input_result']['image']  // object, source image
                , $dest['x']                        // int, destination x
                , $dest['y']                        // int, destination y
                , $src['x']                         // int, source x
                , $src['y']                         // int, source y
                , $dest['w']                        // int, destination width
                , $dest['h']                        // int, destination height
                , $src['w']                         // int, source width
                , $src['h']                         // int, source height
                ))
            { trigger_error(__METHOD__." ERROR: imagecopyresampled() failed ", E_USER_NOTICE); return false; }

        // final check and return
        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: failed to create destination image resource ", E_USER_NOTICE); return false; }

        $result = array(
            'image'    => $image
            , 'width'  => imagesx($image)
            , 'height' => imagesy($image)
            );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Extract a width x height portion of the image, starting from the top,left position
     *
     * @param array $params
     * - REQUIRED
     * - input_result['image']  : (object) input image
     * - REQUIRED - one of the following
     * - transform['top']       : (integer) pixels
     * - transform['left']      : (integer) pixels
     * - transform['width']     : (integer) pixels
     * - transform['height']    : (integer) pixels
     *
     * @return mixed    : false for error, or image info array
     * - image  : (object)
     * - width  : (integer)
     * - height : (integer)
     */
    protected function transform_crop( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        if (!$this->is_image($params['input_result']['image']))
            { trigger_error(__METHOD__." ERROR: missing input image resource ", E_USER_NOTICE); return false; }

        if (empty($params['transform']['top']) && empty($params['transform']['left']) && empty($params['transform']['width']) && empty($params['transform']['height']))
            { trigger_error(__METHOD__." ERROR: missing transform param 'top', 'left', 'width' or 'height' ", E_USER_NOTICE); return false; }

        // source rectangle default
        $src = array(
            'x'   => 0
            , 'y' => 0
            , 'w' => $params['input_result']['width']
            , 'h' => $params['input_result']['height']
            );
        // destination rectangle default
        $dest = array(
            'x'   => 0
            , 'y' => 0
            , 'w' => $params['input_result']['width']
            , 'h' => $params['input_result']['height']
            );

        // set requested destination rectangle
        if (!empty($params['transform']['top'])) {
            $src['y'] += $params['transform']['top'];
            $src['h'] -= $params['transform']['top'];
        }
        if (!empty($params['transform']['left'])) {
            $src['x'] += $params['transform']['left'];
            $src['w'] -= $params['transform']['left'];
        }
        if (!empty($params['transform']['height'])) {
            $src['h'] = min($params['transform']['height'], $src['h']);
        }
        if (!empty($params['transform']['width'])) {
            $src['w'] = min($params['transform']['width'], $src['w']);
        }

        $dest['w'] = $src['w'];
        $dest['h'] = $src['h'];
$crop_params = $params['transform'];
if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('crop_params','src','dest'),true), E_USER_NOTICE); }

        // get output image resource
        $image = @imagecreatetruecolor($dest['w'],$dest['h']);
        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: failed to create destination image resource ", E_USER_NOTICE); return false; }

        // allow transparency in destination image
        imagealphablending($image,true);

        // copy input -> output
        if (!imagecopyresampled(
                $image                              // object, destination image
                , $params['input_result']['image']  // object, source image
                , $dest['x']                        // int, destination x
                , $dest['y']                        // int, destination y
                , $src['x']                         // int, source x
                , $src['y']                         // int, source y
                , $dest['w']                        // int, destination width
                , $dest['h']                        // int, destination height
                , $src['w']                         // int, source width
                , $src['h']                         // int, source height
                ))
            { trigger_error(__METHOD__." ERROR: imagecopyresampled() failed ", E_USER_NOTICE); return false; }

        // final check and return
        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: failed to create destination image resource ", E_USER_NOTICE); return false; }

        $result = array(
            'image'    => $image
            , 'width'  => imagesx($image)
            , 'height' => imagesy($image)
            );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Resize the image canvas by removing pixels along each edge.
     *
     * @param array $params
     * - REQUIRED
     * - input_result['image']  : (object) input image
     * - REQUIRED - one of the following
     * - transform['top']       : (integer) pixels
     * - transform['left']      : (integer) pixels
     * - transform['bottom']    : (integer) pixels
     * - transform['right']     : (integer) pixels
     *
     * @return mixed    : false for error, or image info array
     * - image  : (object)
     * - width  : (integer)
     * - height : (integer)
     */
    protected function transform_trim( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        if (!$this->is_image($params['input_result']['image']))
            { trigger_error(__METHOD__." ERROR: missing input image resource ", E_USER_NOTICE); return false; }

        if (empty($params['transform']['top']) && empty($params['transform']['left']) && empty($params['transform']['bottom']) && empty($params['transform']['right']))
            { trigger_error(__METHOD__." ERROR: missing transform param 'top', 'left', 'bottom' or 'right' ", E_USER_NOTICE); return false; }

        // source rectangle default
        $src = array(
            'x'   => 0
            , 'y' => 0
            , 'w' => $params['input_result']['width']
            , 'h' => $params['input_result']['height']
            );
        // destination rectangle default
        $dest = array(
            'x'   => 0
            , 'y' => 0
            , 'w' => $params['input_result']['width']
            , 'h' => $params['input_result']['height']
            );

        // set requested destination rectangle
        if (!empty($params['transform']['top'])) {
            $src['y'] += $params['transform']['top'];
            $src['h'] -= $params['transform']['top'];
        }
        if (!empty($params['transform']['left'])) {
            $src['x'] += $params['transform']['left'];
            $src['w'] -= $params['transform']['left'];
        }
        if (!empty($params['transform']['bottom'])) {
            $src['h'] -= $params['transform']['bottom'];
        }
        if (!empty($params['transform']['right'])) {
            $src['w'] -= $params['transform']['right'];
        }

        $dest['w'] = $src['w'];
        $dest['h'] = $src['h'];
//$trim_params = $params['transform'];
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('trim_params','src','dest'),true), E_USER_NOTICE); }

        // get output image resource
        $image = @imagecreatetruecolor($dest['w'],$dest['h']);
        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: failed to create destination image resource ", E_USER_NOTICE); return false; }

        // allow transparency in destination image
        imagealphablending($image,true);

        // copy input -> output
        if (!imagecopyresampled(
                $image                              // object, destination image
                , $params['input_result']['image']  // object, source image
                , $dest['x']                        // int, destination x
                , $dest['y']                        // int, destination y
                , $src['x']                         // int, source x
                , $src['y']                         // int, source y
                , $dest['w']                        // int, destination width
                , $dest['h']                        // int, destination height
                , $src['w']                         // int, source width
                , $src['h']                         // int, source height
                ))
            { trigger_error(__METHOD__." ERROR: imagecopyresampled() failed ", E_USER_NOTICE); return false; }

        // final check and return
        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: failed to create destination image resource ", E_USER_NOTICE); return false; }

        $result = array(
            'image'    => $image
            , 'width'  => imagesx($image)
            , 'height' => imagesy($image)
            );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Resize the image canvas by adding pixels along each edge.
     *
     * @param array $params
     * - REQUIRED
     * - input_result['image']  : (object) input image
     * - REQUIRED - one of the following
     * - transform['top']       : (integer) pixels
     * - transform['left']      : (integer) pixels
     * - transform['bottom']    : (integer) pixels
     * - transform['right']     : (integer) pixels
     * - OPTIONAL
     * - transform['color_r'] : (integer) bgcolor 0..255
     * - transform['color_g'] : (integer) bgcolor 0..255
     * - transform['color_b'] : (integer) bgcolor 0..255
     *
     * @return mixed    : false for error, or image info array
     * - image  : (object)
     * - width  : (integer)
     * - height : (integer)
     */
    protected function transform_border( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        if (!$this->is_image($params['input_result']['image']))
            { trigger_error(__METHOD__." ERROR: missing input image resource ", E_USER_NOTICE); return false; }

        if (empty($params['transform']['top']) && empty($params['transform']['left']) && empty($params['transform']['bottom']) && empty($params['transform']['right']))
            { trigger_error(__METHOD__." ERROR: missing transform param 'top', 'left', 'bottom' or 'right' ", E_USER_NOTICE); return false; }

        // source rectangle default
        $src = array(
            'x'   => 0
            , 'y' => 0
            , 'w' => $params['input_result']['width']
            , 'h' => $params['input_result']['height']
            );
        // destination rectangle default
        $dest = array(
            'x'   => 0
            , 'y' => 0
            , 'w' => $params['input_result']['width']
            , 'h' => $params['input_result']['height']
            );

        // set requested destination rectangle
        $new_w = $dest['w'];
        $new_h = $dest['h'];
        if (!empty($params['transform']['top'])) {
            $dest['y'] += $params['transform']['top'];
            $new_h += $params['transform']['top'];
        }
        if (!empty($params['transform']['left'])) {
            $dest['x'] += $params['transform']['left'];
            $new_w += $params['transform']['left'];
        }
        if (!empty($params['transform']['bottom'])) {
            $new_h += $params['transform']['bottom'];
        }
        if (!empty($params['transform']['right'])) {
            $new_w += $params['transform']['right'];
        }

        $color_r = abs($params['transform']['color_r'] ?? 0);
        $color_r = min( abs($color_r), 255 );

        $color_g = abs($params['transform']['color_g'] ?? 0);
        $color_g = min( abs($color_g), 255 );

        $color_b = abs($params['transform']['color_b'] ?? 0);
        $color_b = min( abs($color_b), 255 );
//$border_params = $params['transform'];
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('border_params','src','dest','new_w','new_h'),true), E_USER_NOTICE); }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('color_r','color_g','color_b'),true), E_USER_NOTICE); }

        // get output image resource
        $image = @imagecreatetruecolor($new_w,$new_h);
        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: failed to create destination image resource ", E_USER_NOTICE); return false; }

        // border color fill
        $border_color = imagecolorallocate($image, $color_r, $color_g, $color_b);
        imagefill($image, 0, 0, $border_color);

        // copy input -> output
        if (!imagecopyresampled(
                $image                              // object, destination image
                , $params['input_result']['image']  // object, source image
                , $dest['x']                        // int, destination x
                , $dest['y']                        // int, destination y
                , $src['x']                         // int, source x
                , $src['y']                         // int, source y
                , $dest['w']                        // int, destination width
                , $dest['h']                        // int, destination height
                , $src['w']                         // int, source width
                , $src['h']                         // int, source height
                ))
            { trigger_error(__METHOD__." ERROR: imagecopyresampled() failed ", E_USER_NOTICE); return false; }

        // final check and return
        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: failed to create destination image resource ", E_USER_NOTICE); return false; }

        $result = array(
            'image'    => $image
            , 'width'  => imagesx($image)
            , 'height' => imagesy($image)
            );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Flip the image horizontally (x-axis) or vertically (y-axis).
     *
     * @param array $params
     * - REQUIRED
     * - input_result['image']  : (object) input image
     * - transform['axis']      : (string) "x" || "y"
     *
     * @return mixed    : false for error, or image info array
     * - image  : (object)
     * - width  : (integer)
     * - height : (integer)
     */
    protected function transform_flip( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        if (!$this->is_image($params['input_result']['image']))
            { trigger_error(__METHOD__." ERROR: missing input image resource ", E_USER_NOTICE); return false; }

        if (empty($params['transform']['axis']))
            { trigger_error(__METHOD__." ERROR: missing transform param 'axis' ", E_USER_NOTICE); return false; }

        // source rectangle default
        $src = array(
            'x'   => ($params['transform']['axis'] == 'x') ? $params['input_result']['width'] - 1 : 0
            , 'y' => ($params['transform']['axis'] == 'x') ? 0 : $params['input_result']['height'] - 1
            , 'w' => ($params['transform']['axis'] == 'x') ? 0 - $params['input_result']['width'] : $params['input_result']['width']
            , 'h' => ($params['transform']['axis'] == 'x') ? $params['input_result']['height'] : 0 - $params['input_result']['height']
            );
        // destination rectangle default
        $dest = array(
            'x'   => 0
            , 'y' => 0
            , 'w' => $params['input_result']['width']
            , 'h' => $params['input_result']['height']
            );
//$flip_params = $params['transform'];
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('flip_params','src','dest'),true), E_USER_NOTICE); }

        // get output image resource
        $image = @imagecreatetruecolor($dest['w'],$dest['h']);
        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: failed to create destination image resource ", E_USER_NOTICE); return false; }

        // allow transparency in destination image
        imagealphablending($image,true);

        // copy input -> output
        if (!imagecopyresampled(
                $image                              // object, destination image
                , $params['input_result']['image']  // object, source image
                , $dest['x']                        // int, destination x
                , $dest['y']                        // int, destination y
                , $src['x']                         // int, source x
                , $src['y']                         // int, source y
                , $dest['w']                        // int, destination width
                , $dest['h']                        // int, destination height
                , $src['w']                         // int, source width
                , $src['h']                         // int, source height
                ))
            { trigger_error(__METHOD__." ERROR: imagecopyresampled() failed ", E_USER_NOTICE); return false; }

        // final check and return
        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: failed to create destination image resource ", E_USER_NOTICE); return false; }

        $result = array(
            'image'    => $image
            , 'width'  => imagesx($image)
            , 'height' => imagesy($image)
            );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Rotate the image clockwise.
     *
     * @param array $params
     * - REQUIRED
     * - input_result['image']  : (object) input image
     * - transform['degrees']   : (integer) clockwise degrees from vertical
     * - OPTIONAL
     * - transform['color_r'] : (integer) bgcolor 0..255
     * - transform['color_g'] : (integer) bgcolor 0..255
     * - transform['color_b'] : (integer) bgcolor 0..255
     *
     * @return mixed    : false for error, or image info array
     * - image  : (object)
     * - width  : (integer)
     * - height : (integer)
     */
    protected function transform_rotate( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        if (!$this->is_image($params['input_result']['image']))
            { trigger_error(__METHOD__." ERROR: missing input image resource ", E_USER_NOTICE); return false; }

        if (empty($params['transform']['degrees']))
            { trigger_error(__METHOD__." ERROR: missing transform param 'degrees' ", E_USER_NOTICE); return false; }

        $degrees = intval($params['transform']['degrees'] ?? 0);
        $degrees = intval(360 - ($degrees % 360));  // clockwise (not counter-clockwise)

        $color_r = abs($params['transform']['color_r'] ?? 0);
        $color_r = min( abs($color_r), 255 );

        $color_g = abs($params['transform']['color_g'] ?? 0);
        $color_g = min( abs($color_g), 255 );

        $color_b = abs($params['transform']['color_b'] ?? 0);
        $color_b = min( abs($color_b), 255 );

//$rotate_params = $params['transform'];
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('rotate_params','degrees','color_r','color_g','color_b'),true), E_USER_NOTICE); }

        // get output image resource
        $image = $params['input_result']['image'];
        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: failed to create destination image resource ", E_USER_NOTICE); return false; }

        $bgcolor = imagecolorallocate($image, $color_r, $color_g, $color_b);
        $image = imagerotate($image, $degrees, $bgcolor);

        // final check and return
        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: failed to create destination image resource ", E_USER_NOTICE); return false; }

        $result = array(
            'image'    => $image
            , 'width'  => imagesx($image)
            , 'height' => imagesy($image)
            );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Resize image proportionally
     *
     * @param array $params
     * - REQUIRED
     * - input_result['image']  : (object) input image
     * - transform['percent']   : (integer) proportional dimension of input image
     *
     * @return mixed    : false for error, or image info array
     * - image  : (object)
     * - width  : (integer)
     * - height : (integer)
     */
    protected function transform_scale( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        if (!$this->is_image($params['input_result']['image']))
            { trigger_error(__METHOD__." ERROR: missing input image resource ", E_USER_NOTICE); return false; }
//         if (empty($params['transform']) || !is_array($params['transform']))
//             { trigger_error(__METHOD__." ERROR: missing param array 'transform' ", E_USER_NOTICE); return false; }
        if (empty($params['transform']['percent']))
            { trigger_error(__METHOD__." ERROR: missing transform param 'percent' ", E_USER_NOTICE); return false; }

        // source rectangle default
        $src = array(
            'x'   => 0
            , 'y' => 0
            , 'w' => $params['input_result']['width']
            , 'h' => $params['input_result']['height']
            );
        // destination rectangle default
        $dest = array(
            'x'   => 0
            , 'y' => 0
            , 'w' => $params['input_result']['width']
            , 'h' => $params['input_result']['height']
            );

        // set requested destination rectangle
        $percent = intval($params['transform']['percent'] ?? 100);

        $dest['w'] = floor($src['w'] * ($percent/100));
        $dest['h'] = floor($src['h'] * ($percent/100));
//$scale_params = $params['transform'];
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('scale_params','src','dest'),true), E_USER_NOTICE); }

        // get output image resource
        $image = @imagecreatetruecolor($dest['w'],$dest['h']);
        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: failed to create destination image resource ", E_USER_NOTICE); return false; }

        // allow transparency in destination image
        imagealphablending($image,true);

        // copy input -> output
        if (!imagecopyresampled(
                $image                              // object, destination image
                , $params['input_result']['image']  // object, source image
                , $dest['x']                        // int, destination x
                , $dest['y']                        // int, destination y
                , $src['x']                         // int, source x
                , $src['y']                         // int, source y
                , $dest['w']                        // int, destination width
                , $dest['h']                        // int, destination height
                , $src['w']                         // int, source width
                , $src['h']                         // int, source height
                ))
            { trigger_error(__METHOD__." ERROR: imagecopyresampled() failed ", E_USER_NOTICE); return false; }

        // final check and return
        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: failed to create destination image resource ", E_USER_NOTICE); return false; }

        $result = array(
            'image'    => $image
            , 'width'  => imagesx($image)
            , 'height' => imagesy($image)
            );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Resize and crop image proportionally
     *
     * @param array $params
     * - REQUIRED
     * - input_result['image']  : (object) input image
     * - transform['percent']   : (integer) proportional dimension of input image
     * - OPTIONAL
     * - transform['width']  : (integer) default = input width
     * - transform['height'] : (integer) default = input height
     * - transform['focus']  : (integer) default = X, center
     *
     * @return mixed    : false for error, or image info array
     * - image  : (object)
     * - width  : (integer)
     * - height : (integer)
     */
    protected function transform_zoom( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        if (!$this->is_image($params['input_result']['image']))
            { trigger_error(__METHOD__." ERROR: missing input image resource ", E_USER_NOTICE); return false; }

        if (empty($params['transform']['percent']))
            { trigger_error(__METHOD__." ERROR: missing transform param 'percent' ", E_USER_NOTICE); return false; }

        // source rectangle default
        $src = array(
            'x'   => 0
            , 'y' => 0
            , 'w' => $params['input_result']['width']
            , 'h' => $params['input_result']['height']
            );
        // destination rectangle default
        $dest = array(
            'x'   => 0
            , 'y' => 0
            , 'w' => intval($params['transform']['width']  ?? 0)
            , 'h' => intval($params['transform']['height'] ?? 0)
            );

        // get full size of zoomed image
        $percent = intval($params['transform']['percent'] ?? 100);
        $focus   = strtolower($params['transform']['focus'] ?? 'x');

        $start_width  = $params['input_result']['width'];
        $start_height = $params['input_result']['height'];
        $full_width   = floor($start_width  * ($percent/100));
        $full_height  = floor($start_height * ($percent/100));

        // set destination
        if (empty($dest['w']) && empty($dest['h'])) {
            $dest['w'] = $start_width;
            $dest['h'] = $start_height;
        } else if (!empty($dest['w']) && empty($dest['h'])) {
            $dest['h'] = floor(($full_height / $full_width) * $dest['w']);
        } else if (empty($dest['w']) && !empty($dest['h'])) {
            $dest['w'] = floor(($full_width / $full_height) * $dest['h']);
        }
        $dest['w'] = min($dest['w'], $full_width);
        $dest['h'] = min($dest['h'], $full_height);

        // set source
        $src['w'] = floor(($dest['w'] / $full_width)  * $start_width);
        $src['h'] = floor(($dest['h'] / $full_height) * $start_height);

        switch ($focus)
        {
            case 'n':
            case 'north':
                $src['x'] = floor(($start_width  - $src['w']) / 2);
                $src['y'] = 0;
                break;
            case 's':
            case 'south':
                $src['x'] = floor(($start_width  - $src['w']) / 2);
                $src['y'] = $start_height - $src['h'];
                break;
            case 'e':
            case 'east':
                $src['x'] = $start_width  - $src['w'];
                $src['y'] = floor(($start_height - $src['h']) / 2);
                break;
            case 'w':
            case 'west':
                $src['x'] = 0;
                $src['y'] = floor(($start_height - $src['h']) / 2);
                break;

            case 'ne':
            case 'northeast':
                $src['x'] = $start_width  - $src['w'];
                $src['y'] = 0;
                break;
            case 'nw':
            case 'northwest':
                $src['x'] = 0;
                $src['y'] = 0;
                break;
             case 'se':
            case 'southeast':
                $src['x'] = $start_width  - $src['w'];
                $src['y'] = $start_height - $src['h'];
                break;
           case 'sw':
            case 'southwest':
                $src['x'] = 0;
                $src['y'] = $start_height - $src['h'];
                break;

            case 'x':
            case 'center':
            default:
                $src['x'] = floor(($start_width  - $src['w']) / 2);
                $src['y'] = floor(($start_height - $src['h']) / 2);
                break;
        }
// $zoom_params = $params['transform'];
// if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('zoom_params','src','dest'),true), E_USER_NOTICE); }

        // get output image resource
        $image = @imagecreatetruecolor($dest['w'],$dest['h']);
        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: failed to create destination image resource ", E_USER_NOTICE); return false; }

        // allow transparency in destination image
        imagealphablending($image,true);

        // copy input -> output
        if (!imagecopyresampled(
                $image                              // object, destination image
                , $params['input_result']['image']  // object, source image
                , $dest['x']                        // int, destination x
                , $dest['y']                        // int, destination y
                , $src['x']                         // int, source x
                , $src['y']                         // int, source y
                , $dest['w']                        // int, destination width
                , $dest['h']                        // int, destination height
                , $src['w']                         // int, source width
                , $src['h']                         // int, source height
                ))
            { trigger_error(__METHOD__." ERROR: imagecopyresampled() failed ", E_USER_NOTICE); return false; }

        // final check and return
        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: failed to create destination image resource ", E_USER_NOTICE); return false; }

        $result = array(
            'image'    => $image
            , 'width'  => imagesx($image)
            , 'height' => imagesy($image)
            );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }


}  // end class IMG_Processor_GD
