<?php
/**
 * Image proccessor
 *
 * @package           tenandtwo-wp-plugins
 * @subpackage        tenandtwo-img-processor
 * @author            Ten & Two Systems
 * @copyright         2024 Ten & Two Systems
 */


/**
 * Exstensible for additional "transform" params
 *
 * @see IMG_Processor_Imagick::getImageCache()
 */
class IMG_Processor_Imagick
{

    /**
     * @var array Imagick package info
     */
    public $imagick_info;
    /**
     * @var array returned by Imagick::queryFormats()
     */
    public $imagick_types;

    /**
     * init members
     * cleanCache() per gc_probability
     */
    function __construct()
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }
//if (WP_DEBUG) { echo "<pre>"; debug_print_backtrace(); echo "</pre>\n"; }

        $this->imagick_info  = null;
        $this->imagick_types = null;  // GIF | JPEG | PNG | WEBP ...

        if (ini_get('session.gc_probability') >= rand(1,ini_get('session.gc_divisor')))
            { $cache = $this->cleanCache(); }
    }

    function __destruct()
    {
//if (WP_DEBUG) { trigger_error(__METHOD__, E_USER_NOTICE); }
    }

    /**
     * Check image resource/class
     */
    public function is_image( $image )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('image'),true), E_USER_NOTICE); }
        return ($image instanceof Imagick);
    }

    /**
     * Populate this->imagick_info = array()
     *
     * <code>
     * this->imagick_info = array(
     *     'package'    => {string} package name
     *     , 'version'  => {string} dotted version number
     *     , 'release'  => {string} package date
     *     );
     * </code>
     *
     * @param none
     * @return bool           true for success ; false for error
     */
    public function initHandler()
    {
//if (WP_DEBUG) { trigger_error(__METHOD__, E_USER_NOTICE); }
        if (is_array($this->imagick_info) && count($this->imagick_info))
            { return true; }
        if (!IMG_PROCESSOR_IMAGICK)
            { trigger_error(__METHOD__." ERROR: Imagick extension not available. ", E_USER_NOTICE); return false; }

        $this->imagick_types = Imagick::queryFormats();
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(array('imagick_types' => $this->imagick_types),true), E_USER_NOTICE); }

        $this->imagick_info = array(
            'package' => Imagick::getPackageName(),
            'version' => '',
            'release' => Imagick::getReleaseDate(),
            );
//Imagick::getResourceLimit();  // RESOURCETYPE_UNDEFINED, RESOURCETYPE_AREA, RESOURCETYPE_DISK, RESOURCETYPE_FILE, RESOURCETYPE_MAP, RESOURCETYPE_MEMORY

        $v = Imagick::getVersion();
        preg_match('/ImageMagick ([0-9]+\.[0-9]+\.[0-9]+)/', $v['versionString'], $v);
        $this->imagick_info['version'] = $v[1] ?? 0;
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(array('imagick_info' => $this->imagick_info),true), E_USER_NOTICE); }

        return true;
    }

    /**
     * Get info for existing image file
     * Supported 'file' protocols: https://www.php.net/manual/en/wrappers.php
     *
     * @param array $params     required: file
     * - params['file'] : (string) local absolute filepath or supported remote url
     * - params['alt']  : (string) img alt attribute, default to basename
     *
     * @return mixed  : false for error, or array
     * - file         : (string)  local filepath or remote url
     * - width        : (integer) pixels
     * - height       : (integer) pixels
     * - type         : (string)  from this->imagick_types
     * - content_type : (string)  for output
     * - readable     : (bool)
     * - writeable    : (bool)
     * - bytes        : (integer) filesize                          (123456)
     * - resolution   : (string)  DPI                               (72,72)
     * - datetime     : (string)  modificaiton date                 (2024-01-01 12:00:00)
     * - dir          : (string)  absolute path to image directory  (/www/.../html/pics/)
     * - basename     : (string)  filename with extension           (mypic.jpg)
     * - name         : (string)  filename without extension        (mypic)
     * - ext          : (string)  filename extension                (.jpg)
     * - url          : (string)  local path or remote url
     * - alt          : (string)  original file name
     * - img          : (string)  default <img/> tag
     */
    public function getImageFileInfo( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        if (!$this->initHandler()) { return false; }

        if (empty($params['file']))
            { trigger_error(__METHOD__." ERROR: missing param 'file' ", E_USER_NOTICE); return false; }

        $exists = false;
        $remote = filter_var($params['file'], FILTER_VALIDATE_URL);
        $headers = null;
        if ($remote) {
            $headers = @get_headers($params['file'], 1);
            $headers = array_change_key_case($headers,CASE_LOWER);
            $exists = (!empty($headers[0]) && strpos($headers[0],'404') === false);
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('headers','exists'),true), E_USER_NOTICE); }
        } else {
            $exists = is_file($params['file']);
        }
        if (!$exists)
            { trigger_error(__METHOD__." ERROR: Invalid path ; file does not exist.  (".$params['file'].") ", E_USER_NOTICE); return false; }

        //$bytes = $image->getImageLength();
        $bytes = ($remote)
            ? ((strcasecmp($headers[0], 'HTTP/1.1 200 OK') != 0) ? $headers['content-length'][1] : $headers['content-length'])
            : filesize($params['file']);

        $image = new Imagick( $params['file'] );
        $width        = $image->getImageWidth();
        $height       = $image->getImageHeight();
        $type         = $image->getImageFormat();
        $content_type = $image->getImageMimeType();
        $readable     = in_array($type,$this->imagick_types) && ($remote || is_readable($params['file']));
        $writeable    = !$remote && is_writeable($params['file']);
        $resolution   = $image->getImageResolution();
        $datetime     = date("Y-m-d H:i:s", @filemtime($params['file']) ?: time());

        $pathinfo = pathinfo($params['file']);  // get 'dirname', 'basename', 'extension', 'filename'
        $img_alt = $params['alt'] ?? $pathinfo['basename'];

        if ($remote) {
            $url = $params['file'];
        } elseif (strpos($params['file'],ABSPATH) !== false) {
            $url = strtr($params['file'], array(ABSPATH => '/'));
        }
        $img_tag = ($url) ? '<img src="' . $url . '" width="' . $width . '" height="' . $height . '" alt="' . $img_alt . '" />' : '';

        $result = array(
            'img'            => $img_tag
            , 'file'         => $params['file']
            , 'url'          => $url
            , 'width'        => $width
            , 'height'       => $height
            , 'type'         => $type
            , 'content_type' => $content_type
            , 'readable'     => $readable
            , 'writeable'    => $writeable
            , 'bytes'        => $bytes
            , 'resolution'   => $resolution['x'].','.$resolution['y']
            , 'datetime'     => $datetime
            , 'dir'          => $pathinfo['dirname'].'/'
            , 'basename'     => $pathinfo['basename']
            , 'name'         => $pathinfo['filename']
            , 'ext'          => $pathinfo['extension']
            , 'alt'          => $img_alt
            );

//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }
if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Same as getImage(), with results cached at options['cache_path'] or 'dir' param
     *
     * @uses IMG_Processor_Imagick::getImage()
     *
     * @param array $params
     * - input         : (array) see getImage()
     * - transform     : (array) see getImage()
     * - output        : (array) see getImage()
     * - output['file'] : (string)  override cache dir+name+type for getImage()
     * - output['type'] : (string)  default = input image type
     * - output['dpi']  : (integer) default = input image resolution
     * - output['nocache'] : (bool)
     *
     * @return bool             false for error, or array image file info
     */
    public function getImageCache( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        $options = get_option( IMG_PROCESSOR_OPTS, array() );

        if (empty($params['input']))
            { trigger_error(__METHOD__." ERROR: missing param array 'input' ", E_USER_NOTICE); return false; }
        if (empty($params['transform']))
            { $params['transform'] = array(); }
        if (empty($params['output']))
            { $params['output'] = array(); }

        // get source image info
        $info = $this->getImageFileInfo( $params['input'] );
        if (!$info) { return false; }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('info'),true), E_USER_NOTICE); }
        $info['_usage'] = IMG_Processor_Util::getImageCommands( $params, $info );

        // no transform, no cache ; return source info
        if (empty($params['transform']['type']) || $params['transform']['type'] == 'none')
            { return $info; }

        $output_type = $info['type']; // original vs pref vs param
        if (!empty($options['output_type']) && in_array(strtoupper($options['output_type']),$this->imagick_types))
            { $output_type = $options['output_type']; }
        if (!empty($params['output']['type']) && in_array(strtoupper($params['output']['type']),$this->imagick_types))
            { $output_type = $params['output']['type']; }

        $output_dpi = 0;             // original vs pref vs param
        if (!empty($options['output_dpi']))
            { $output_dpi = $options['output_dpi']; }
        if (isset($params['output']['dpi']) && intval($params['output']['dpi']) >= 0)
            { $output_dpi = $params['output']['dpi']; }

        $params['output']['nocache'] = !empty($params['output']['nocache']);

        // default cache location
        $params['output']['dir']  = rtrim($options['cache_path'],'/').'/';
        $params['output']['name'] = null;
        $params['output']['type'] = $output_type;
        $params['output']['dpi']  = intval($output_dpi);
        // override cache location
        if (!empty($params['output']['file']))
        {
            $pathinfo = pathinfo($params['output']['file']);  // get 'dirname', 'basename', 'extension', 'filename'
            $params['output']['dir']  = rtrim($pathinfo['dirname'],'/').'/';
            $params['output']['name'] = $pathinfo['filename'];
            $params['output']['type'] = $pathinfo['extension'];
        }

        // output name
        if (empty($params['output']['nocache']) && empty($params['output']['file']))
        {
            // get output directory + name + type
            $params['output']['dir'] .= IMG_Processor_Util::getHash(array('data' => $info['dir'], 'method' => 'crc32b')) . '/';
            $params['output']['dir'] .= strtr($info['name'].'_'.strtolower($info['type']), array('.' => '_')) . '/';
            //$params['output']['dir'] .= strtr($info['name'].'_'.$info['ext'], array('.' => '_')) . '/';

            // get output filename
            switch($params['transform']['type']) {
                case 'fit':
                case 'fill':
                case 'resize':
                    $params['output']['name'] = $params['transform']['type']
                        . '_' . ($params['transform']['width']  ?? '0')
                        . '_' . ($params['transform']['height'] ?? '0')
                        . '';
                    break;
                case 'crop':
                    $params['output']['name'] = $params['transform']['type']
                        . '_' . ($params['transform']['top']    ?? '0')
                        . '_' . ($params['transform']['left']   ?? '0')
                        . '_' . ($params['transform']['width']  ?? '0')
                        . '_' . ($params['transform']['height'] ?? '0')
                        . '';
                    break;
                case 'trim':
                    $params['output']['name'] = $params['transform']['type']
                        . '_' . ($params['transform']['top']    ?? '0')
                        . '_' . ($params['transform']['left']   ?? '0')
                        . '_' . ($params['transform']['bottom'] ?? '0')
                        . '_' . ($params['transform']['right']  ?? '0')
                        . '';
                    break;
                case 'border':
                    $params['output']['name'] = $params['transform']['type']
                        . '_' . ($params['transform']['top']    ?? '0')
                        . '_' . ($params['transform']['left']   ?? '0')
                        . '_' . ($params['transform']['bottom'] ?? '0')
                        . '_' . ($params['transform']['right']  ?? '0')
                        . '_' . sprintf("%02x%02x%02x%02x", ($params['transform']['color_r'] ?? '0'), ($params['transform']['color_g'] ?? '0'), ($params['transform']['color_b'] ?? '0')
                                , (isset($params['transform']['color_a']) ? ($params['transform']['color_a'] * 10) : '0'))
                        . '';
                    break;
                case 'flip':
                    $params['transform']['axis'] = $params['transform']['axis'] ?? 'y';
                    $params['output']['name'] = $params['transform']['type']
                        . '_' . (($params['transform']['axis'] == 'x') ? 'x' : 'y')
                        . '';
                    break;
                case 'rotate':
                    $params['transform']['degrees'] = $params['transform']['degrees'] % 360 ?? '0';
                    if ($params['transform']['degrees'] < 0)
                        { $params['transform']['degrees'] = 360 + $params['transform']['degrees']; }
                    $params['output']['name'] = $params['transform']['type']
                        . '_' . ($params['transform']['degrees'] ?? '0')
                        . '_' . sprintf("%02x%02x%02x%02x", ($params['transform']['color_r'] ?? '0'), ($params['transform']['color_g'] ?? '0'), ($params['transform']['color_b'] ?? '0')
                                , (isset($params['transform']['color_a']) ? ($params['transform']['color_a'] * 10) : '0'))
                        . '';
                    break;
                case 'scale':
                    $params['output']['name'] = $params['transform']['type']
                        . '_' . ($params['transform']['percent'] ?? '0')
                        . '';
                    break;
                case 'zoom':
                    $params['output']['name'] = $params['transform']['type']
                        . '_' . ($params['transform']['percent'] ?? '0')
                        . '_' . strtolower($params['transform']['focus'] ?? 'x')
                        . '_' . ($params['transform']['width']   ?? '0')
                        . '_' . ($params['transform']['height']  ?? '0')
                        . '';
                    break;

                case 'none':
                default:
                    $params['output']['name'] = $info['name'];
                    $params['output']['nocache'] = true;
                    break;
            }
        }

        // check for cached result
        if (empty($params['output']['nocache']))
        {
            // setup/check output cache directory
            $cache_file = $params['output']['dir'] . strtolower($params['output']['name'].'.'.$params['output']['type']);
            $cache_exists = false;

            clearstatcache();
            if (!is_dir($params['output']['dir'])) {
                umask(0000);
                if (!mkdir($params['output']['dir'], 0775, true))
                    { trigger_error(__METHOD__." ERROR: could not create folder '".$params['output']['dir']."' ", E_USER_NOTICE); return false; }
            } else {
                $cache_exists = is_file($cache_file);
            }
if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('cache_file','cache_exists'),true), E_USER_NOTICE); }

            // delete cache file if source updated
            if ($cache_exists && filemtime($cache_file) < @filemtime($info['file'])) {
                $cache_exists = false;
                if (!unlink($cache_file))
                    { trigger_error(__METHOD__." ERROR: could not remove out-dated file '".$cache_file."' ", E_USER_NOTICE); return false; }
            }

            // return existing cache info
            if ($cache_exists) {
                $subparams = array(
                    'file'  => realpath($cache_file),
                    'alt'   => $info['basename'],  // original filename
                    );
                $result = $this->getImageFileInfo( $subparams );
                if (!$result) { return false; }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('cache_exists','params','result'),true), E_USER_NOTICE); }

$result['_usage'] = IMG_Processor_Util::getImageCommands( $params, $result );
                return $result;
            }
        }

        // if nocache, clear output params ; getImage() will return 'data' not 'file'
        if (!empty($params['output']['nocache'])) { $params['output'] = array(); }

        // return new cache info
        $result = $this->getImage( $params );
        if (!$result) { return false; }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('cache_exists','params','result'),true), E_USER_NOTICE); }

$result['_usage'] = IMG_Processor_Util::getImageCommands( $params, $result );
        return $result;
    }

    /**
     * Delete all or unused items in image cache directory
     * remove all || files not created|modified|accessed for expire_value
     *
     * @param array $params
     * - cache_path     : (string) directory path
     * - expire_type    : (string) atime|amin|ctime|cmin|mtime|mmin = [access|create|modify][days|minutes]
     * - expire_value   : (integer) number of expire_units
     *
     * @return mixed    : false for error, or array for cache directory
     */
    public function cleanCache( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        //$valid_expire_type = array('all','Bmin','Btime', 'amin','atime', 'cmin','ctime', 'mmin','mtime',);
        $valid_expire_type = array('all','atime','amin','ctime','cmin','mtime','mmin');

        $options = get_option( IMG_PROCESSOR_OPTS, array() );
        $cache_path   = $params['cache_path']   ?? $options['cache_path'];
        $expire_type  = $params['expire_type']  ?? $options['expire_type'];
        $expire_value = $params['expire_value'] ?? $options['expire_value'];

        if (!is_dir($cache_path))
            { trigger_error(__METHOD__." ERROR: cache directory not found '".$cache_path."' ", E_USER_NOTICE); return false; }
        if (!is_writeable($cache_path))
            { trigger_error(__METHOD__." ERROR: cache directory not writeable '".$cache_path."' ", E_USER_NOTICE); return false; }

        if (!in_array($expire_type,$valid_expire_type))
            { trigger_error(__METHOD__." ERROR: invalid cache expire_type '".$expire_type."' ", E_USER_NOTICE); return false; }
        if (!(0 <= intval($expire_value)))
            { trigger_error(__METHOD__." ERROR: invalid cache expire_value '".$expire_value."' ", E_USER_NOTICE); return false; }

        $cmd_files = "find ".$cache_path."/. -type f -name '*.*'"
                . (($expire_type == "all") ? "" : " -".$expire_type." +".$expire_value)
                . " -delete";
        $cmd_dirs = "find ".$cache_path."/. -depth -type d -empty -delete";

        // run command-line
        @exec($cmd_files . ' ; ' . $cmd_dirs);

        $result = array(
            'cache_path'     => $cache_path
            , 'expire_type'  => $expire_type
            , 'expire_value' => $expire_value
            );

if (WP_DEBUG) { trigger_error(__METHOD__." : exec($cmd_files ; $cmd_dirs) ", E_USER_NOTICE); }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params','result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Read source image, transform, output result
     *
     * @uses IMG_Processor_Imagick::initHandler()
     * @uses IMG_Processor_Imagick::readImage()
     * @uses IMG_Processor_Imagick::transformImage()
     * @uses IMG_Processor_Imagick::writeImage()
     *
     * @param array $params
     * REQUIRED : one of the following
     * - file : (string) absolute filepath or supported remote url
     * - data : (string) dbresult || result of previous getImage() call
     *
     * transform              : (array)
     * - transform['type']    : (string) none || fit || fill || resize || crop || trim || border
     *                          || flip || rotate || scale || zoom
     *
     * if type == fit, one param required
     * - transform['width']   : (integer) pixels
     * - transform['height']  : (integer) pixels
     *
     * if type == resize or type == fill, both params required
     * - transform['width']   : (integer) pixels
     * - transform['height']  : (integer) pixels
     *
     * if type == crop or type == border, one param required
     * - transform['top']     : (integer) pixels
     * - transform['left']    : (integer) pixels
     * - transform['bottom']  : (integer) pixels
     * - transform['right']   : (integer) pixels
     *
     * if type == border, optional
     * - transform['color_r'] : (integer) bgcolor 0..255
     * - transform['color_g'] : (integer) bgcolor 0..255
     * - transform['color_b'] : (integer) bgcolor 0..255
     * - transform['color_a'] : (float) opacity 0..1.0
     *
     * if type == flip, param required
     * - transform['axis'] : (string) x || y
     *
     * if type == rotate, param 'degrees' required
     * - transform['degrees'] : (integer) eg, 90 || 180 || 270
     * - transform['color_r'] : (integer) bgcolor 0..255
     * - transform['color_g'] : (integer) bgcolor 0..255
     * - transform['color_b'] : (integer) bgcolor 0..255
     * - transform['color_a'] : (float) opacity 0..1.0
     *
     * if type == scale, param required
     * - transform['percent'] : (string) eg 50% || 200
     *
     * if type == zoom, percent param required
     * - transform['percent'] : (string) eg 50% || 200
     * - transform['focus']   : (string) X|N|S|E|W|NE|NW|SE|SW
     * - transform['width']   : (integer) pixels
     * - transform['height']  : (integer) pixels
     *
     *
     * output               : (array)
     * - default : output to input dir + name + type
     * - returns raw data if !(force || replace || file || dir || name)
     *
     * output['force']      : (bool)
     * - true = overwrite existing file if necessary
     *
     * output['replace']    : (bool)
     * - true = delete original if writing new file
     *
     * output['dir']        : (string) absolute path to cache directory
     * - default = input file's directory path
     *
     * output['name']       : (string) cache file's basename
     * - default = input file's basename
     *
     * output['type']       : (string) from this->imagick_types
     * - default = input type
     * - determines output .ext
     *
     * output['dpi']        : (integer) resolution
     * - default = input resolution
     *
     * @return bool             false for error, or image info array
     */
    public function getImage( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        if (!$this->initHandler()) { return false; }

        $err = false;
        while (true)
        {
//if (WP_DEBUG) { trigger_error(__METHOD__." : readImage() ", E_USER_NOTICE); }
            $params['input_result']     = $this->readImage( $params );
            if (!$params['input_result'])     { $err = true; break; }
//if (WP_DEBUG) { trigger_error(__METHOD__." : transformImage() ", E_USER_NOTICE); }
            $params['transform_result'] = $this->transformImage( $params );
            if (!$params['transform_result']) { $err = true; break; }
//if (WP_DEBUG) { trigger_error(__METHOD__." : writeImage() ", E_USER_NOTICE); }
            $params['output_result']    = $this->writeImage( $params );
            if (!$params['output_result'])    { $err = true; break; }
            break;
        }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        // free image resources
        if (isset($params['input_result']['image']) && $this->is_image($params['input_result']['image'])) {
            $params['input_result']['image']->clear();
            $params['input_result']['image'] = null;
        }
        if (isset($params['transform_result']['image']) && $this->is_image($params['transform_result']['image'])) {
            $params['transform_result']['image']->clear();
            $params['transform_result']['image'] = null;
        }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        // return false for error
        if ($err) { return false; }

        $result = $params['output_result'];
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Create an image object from existing file or data string
     *
     * @uses IMG_Processor_Imagick::read_image_data()
     * @uses IMG_Processor_Imagick::read_image_file()
     *
     * @param array $params
     * - input['file'] : (string) filepath or supported remote url
     * - input['data'] : (string) dbresult || result of previous getImage() call
     *
     * @return bool             false for error, or image data array
     */
    protected function readImage( $params )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }
        if (empty($params['input']))
            { trigger_error(__METHOD__." ERROR: missing param array 'input' ", E_USER_NOTICE); return false; }

        if (empty($params['input']['file'])) { $params['input']['file'] = null; }
        if (empty($params['input']['data'])) { $params['input']['data'] = null; }
        if (empty($params['input']['file']) && empty($params['input']['data']))
            { trigger_error(__METHOD__." ERROR: missing param 'file' || 'data' ", E_USER_NOTICE); return false; }

        $result = (!empty($params['input']['data']))
                ? $this->read_image_data( $params )
                : $this->read_image_file( $params );
        if (!$result) { return false; }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Dispatch according to params['transform']['type']
     * @see http://www.phpclasses.org/browse/file/25856.html for transform examples
     *
     * @uses IMG_Processor_Imagick::transform_fit()
     * @uses IMG_Processor_Imagick::transform_fill()
     * @uses IMG_Processor_Imagick::transform_resize()
     * @uses IMG_Processor_Imagick::transform_crop()
     * @uses IMG_Processor_Imagick::transform_trim()
     * @uses IMG_Processor_Imagick::transform_border()
     * @uses IMG_Processor_Imagick::transform_flip()
     * @uses IMG_Processor_Imagick::transform_rotate()
     * @uses IMG_Processor_Imagick::transform_scale()
     * @uses IMG_Processor_Imagick::transform_zoom()
     *
     * @param array $params     transform ; output ; input_result
     * - transform['type'] : (string) default == none
     *
     * @return bool             false for error, or transformed image data
     */
    protected function transformImage( $params )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        $result = null;

        if (empty($params['transform']['type']))
            { $params['transform']['type'] = 'none'; }

        switch($params['transform']['type']) {
            case 'none':
                break;

            case 'fit':
                $result = $this->transform_fit( $params );
                break;
            case 'fill':
                $result = $this->transform_fill( $params );
                break;
            case 'resize':
                $result = $this->transform_resize( $params );
                break;
            case 'crop':
                $result = $this->transform_crop( $params );
                break;
            case 'trim':
                $result = $this->transform_trim( $params );
                break;
            case 'border':
                $result = $this->transform_border( $params );
                break;
            case 'flip':
                $result = $this->transform_flip( $params );
                break;
            case 'rotate':
                $result = $this->transform_rotate( $params );
                break;
            case 'scale':
                $result = $this->transform_scale( $params );
                break;
            case 'zoom':
                $result = $this->transform_zoom( $params );
                break;
// filters
            case 'grayscale':

            default:
                trigger_error(__METHOD__." ERROR: unsupported transform type '".$params['transform']['type']."' ", E_USER_NOTICE);
                break;
        }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Dispatch to write_image_data() || write_image_file()
     *
     * @uses IMG_Processor_Imagick::write_image_data()
     * @uses IMG_Processor_Imagick::write_image_file()
     *
     * @param array $params
     * - REQUIRED : one of the following
     * - transform_result   : (array) image object and info
     * - input_result       : (array) image object and info
     * - OPTIONAL
     * - output['force']   : (bool) overwrite existing file if necessary
     * - output['replace'] : (bool) delete original if writing to new file
     * - output['file']    : (string) absolute path to new image file
     * - output['dir']     : (string) path to image file's parent directory
     * - output['name']    : (string) image file's basename
     * - output['type']    : (string) output image type from imagick_types
     *
     * @return bool             false for error, or image info array
     */
    protected function writeImage( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        // output file specs
        if (!isset($params['output']['force']))   { $params['output']['force'] = null; }
        if (!isset($params['output']['replace'])) { $params['output']['replace'] = null; }
        if (!isset($params['output']['file']))    { $params['output']['file'] = null; }
        if (!isset($params['output']['dir']))     { $params['output']['dir'] = null; }
        if (!isset($params['output']['name']))    { $params['output']['name'] = null; }
        if (!isset($params['output']['type']))    { $params['output']['type'] = null; }
        if (!isset($params['output']['dpi']))     { $params['output']['dpi'] = null; }

        if (empty($params['output']['type']) && !empty($params['input_result']['type']))
            { $params['output']['type'] = $params['input_result']['type']; }

        $result = (!($params['output']['force']
                || $params['output']['replace']
                || $params['output']['file']
                || $params['output']['dir']
                || $params['output']['name']
                ))
            ? $this->write_image_data( $params )
            : $this->write_image_file( $params );

//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Create an image object from data string
     *
     * @param array $params
     * - input['data'] : (string) required
     *
     * @return bool             false for error, or image info array
     * - image  : (object) see Imagick::readImageBlob()
     * - width  : (integer)
     * - height : (integer)
     * - type   : (string)
     * - bytes  : (integer)
     */
    protected function read_image_data( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        if (!(isset($params['input']['data']) && strlen($params['input']['data'])))
            { trigger_error(__METHOD__." ERROR: missing param 'data' ", E_USER_NOTICE); return false; }

        $image = new Imagick();
        $image = $image->readImageBlob( $params['input']['data'] );

        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: failed to create image resource for data", E_USER_NOTICE); return false; }

        $result = array(
            'image'     => $image
            , 'width'   => $image->getImageWidth()
            , 'height'  => $image->getImageHeight()
            , 'type'    => $image->getImageFormat()
            , 'bytes'   => $image->getImageLength()
            );

//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Create an image object from existing file
     *
     * @uses IMG_Processor_Imagick::getImageFileInfo()
     *
     * @param array $params
     * - input : (array)
     *
     * @return mixed             false for error, or array
     * - image : (object) see Imagick::readImage()
     */
    protected function read_image_file( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        $info = $this->getImageFileInfo( $params['input'] );
        if (!$info) { return false; }

        if (!strlen($info['file']))
            { trigger_error(__METHOD__." ERROR: missing param 'file' ", E_USER_NOTICE); return false; }
        if (!$info['readable'])
            { trigger_error(__METHOD__." ERROR: unreadable input type '".$info['type']."' ", E_USER_NOTICE); return false; }

        $image = new Imagick( $info['file'] );
        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: failed to create image resource for '".$info['type']."' at '".$info['file']."' ", E_USER_NOTICE); return false; }

        $result = array_merge( array('image' => $image), $info );

//$memory_usage = memory_get_usage(true);
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('memory_usage','result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Returns raw image data in result array
     *
     * @param array $params
     * - REQUIRED
     * - output['type'] : (string) output image type from imagick_types
     * - REQUIRED - one of the following
     * - transform_result   : (array) image object and info
     * - input_result       : (array) image object and info
     *
     * @return mixed            false for error, or image info array
     * - type         : (string) output image_type
     * - width        : (string) output dimension
     * - height       : (string) output dimension
     * - content_type : (string) eg, image/png
     * - data         : (string) image data
     */
    protected function write_image_data( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        // output image
        $image = null;
        if (isset($params['transform_result']['image']) && $this->is_image($params['transform_result']['image']))
            { $image = $params['transform_result']['image']; }
        else if (isset($params['input_result']['image']) && $this->is_image($params['input_result']['image']))
            { $image = $params['input_result']['image']; }
        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: missing param 'transform_result[image]' || 'input_result[image]' ", E_USER_NOTICE); return false; }

        // output image type
        if (empty($params['output']['type']))
            { trigger_error(__METHOD__." ERROR: missing param 'output[type]' ", E_USER_NOTICE); return false; }
        if (!in_array(strtoupper($params['output']['type']),$this->imagick_types))
            { trigger_error(__METHOD__." ERROR: unknown output image type '".$params['output']['type']."' ", E_USER_NOTICE); return false; }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

// resolution?

        // write data
        $image->setImageFormat( strtoupper($params['output']['type']) );
        $data = $image->getImageBlob();

        $result = array(
            'data'           => $data
            , 'width'        => $image->getImageWidth()
            , 'height'       => $image->getImageHeight()
            , 'type'         => $image->getImageFormat()
            , 'bytes'        => $image->getImageLength()
            , 'content_type' => $image->getImageMimeType()
            );

//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Write image data to disk
     *
     * @uses IMG_Processor_Imagick::getImageFileInfo()
     *
     * @param array $params
     * - REQUIRED - one of the following
     * - transform_result   : (array) image object and info
     * - input_result       : (array) image object and info
     * - OPTIONAL
     * - output['force']   : (bool) overwrite existing file if necessary
     * - output['replace'] : (bool) delete original if writing to new file
     * - output['file']    : (string) absolute path to cache file
     * - output['dir']     : (string) absolute path to cache directory
     * - output['name']    : (string) cache file's basename
     * - output['type']    : (string) output image type from imagick_types
     * - output['dpi']     : (integer) output image resolution
     *
     * @return bool             false for error, or image info array
     */
    protected function write_image_file( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        // output image
        $image = null;
        if (isset($params['transform_result']['image']) && $this->is_image($params['transform_result']['image']))
            { $image = $params['transform_result']['image']; }
        else if (isset($params['input_result']['image']) && $this->is_image($params['input_result']['image']))
            { $image = $params['input_result']['image']; }
        if (!$this->is_image($image))
            { trigger_error(__METHOD__." ERROR: missing param 'transform_result[image]' || 'input_result[image]' ", E_USER_NOTICE); return false; }

        $params['output']['force']   = !empty($params['output']['force']);
        $params['output']['replace'] = !empty($params['output']['replace']);

        // output file specs : input file loc used for default
        $params['output']['file'] = trim($params['input_result']['file']);
        $info = pathinfo($params['output']['file']);  // get 'dirname', 'basename', 'extension', 'filename'
        // output file specs : overrides
        if (empty($params['output']['dir']))
            { $params['output']['dir'] = $info['dirname'].'/'; }
        if (empty($params['output']['name']))
            { $params['output']['name'] = $info['filename']; }
        if (empty($params['output']['type']) || !in_array(strtoupper($params['output']['type']),$this->imagick_types))
            { $params['output']['type'] = $info['extension']; }

        // build output filename
        $params['output']['dir']  = rtrim($params['output']['dir'],'/').'/';    // trailing slash
        $params['output']['name'] = strtr($params['output']['name']," ","_");   // no spaces
        $params['output']['type'] = strtolower($params['output']['type']);      // lowercase

        $params['output']['file'] = $params['output']['dir']
                . $params['output']['name']
                . '.'
                . $params['output']['type'];
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        // validate destination path
        if (!is_dir($params['output']['dir']))
            { trigger_error(__METHOD__." ERROR: destination directory '".$params['output']['dir']."' not found ", E_USER_NOTICE); return false; }
        if (!is_writable($params['output']['dir']))
            { trigger_error(__METHOD__." ERROR: destination directory '".$params['output']['dir']."' not writable ", E_USER_NOTICE); return false; }

        // check exists/overwrite
        if (empty($params['output']['force']) && is_file($params['output']['file']))
            { trigger_error(__METHOD__." ERROR: file '".$params['output']['file']."' already exists ", E_USER_NOTICE); return false; }

        // set resolution
        if (!empty($params['output']['dpi']))
        {
            $dpi = $params['output']['dpi'];
            $resolution = $image->getImageResolution();
            if ($resolution['x'] > $dpi || $resolution['y'] > $dpi) {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('dpi','resolution'),true), E_USER_NOTICE); }
                $image->setImageResolution($dpi,$dpi);
                $image->resampleImage($dpi,$dpi,imagick::FILTER_UNDEFINED,1);
            }
        }

        // write file
        //$fspec = $params['output']['type'].':'.$params['output']['file'];
        $fspec = $params['output']['file'];
        if (!$image->writeImage( $fspec ))
            { trigger_error(__METHOD__." ERROR: writeImage('".$fspec."') failed ", E_USER_NOTICE); return false; }

        // return info for new image file
        $subparams = array(
            'file'  => $params['output']['file'],
            'alt'   => $params['input_result']['basename'],  // original filename
            );
        $result = $this->getImageFileInfo( $subparams );
        if (!$result) { return false; }

        // remove old image file, if necessary
        if ($params['output']['replace'] && $params['input_result']['file'] != $params['output']['file']) {
            if (!unlink($params['input_result']['file'])) {
                trigger_error(__METHOD__." ERROR: Could not remove old file '" . $params['input_result']['file'] . "'", E_USER_NOTICE);
                return false;
            }
        }

//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }


    /**
     * Resize image proportionally to fit requested width and/or height
     *
     * @param array $params
     * - REQUIRED
     * - input_result['image']  : (object) input image
     * - REQUIRED - one of the following
     * - transform['width']     : (integer) pixels
     * - transform['height']    : (integer) pixels
     *
     * @return mixed    : false for error, or image info array
     * - image  : (object)
     * - width  : (integer)
     * - height : (integer)
     */
    protected function transform_fit( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        if (!$this->is_image($params['input_result']['image']))
            { trigger_error(__METHOD__." ERROR: missing input image resource ", E_USER_NOTICE); return false; }
        $image = clone $params['input_result']['image'];

        if (empty($params['transform']['width']) && empty($params['transform']['height']))
            { trigger_error(__METHOD__." ERROR: missing transform param 'width' and/or 'height' ", E_USER_NOTICE); return false; }

        $width  = intval($params['transform']['width']  ?? 0);
        $height = intval($params['transform']['height'] ?? 0);

        // transfrom
        $image->resizeImage( $width, $height, Imagick::FILTER_LANCZOS, 1, ($width && $height) );
        //$image->setImagePage( $width, $height, 0, 0 );

        $result = array(
            'image'    => $image
            , 'width'  => $image->getImageWidth()
            , 'height' => $image->getImageHeight()
            );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Resize image proportionally to fill requested width and height
     *
     * @param array $params
     * - REQUIRED
     * - input_result['image']  : (object) input image
     * - REQUIRED - one of the following
     * - transform['width']     : (integer) pixels
     * - transform['height']    : (integer) pixels
     *
     * @return mixed    : false for error, or image info array
     * - image  : (object)
     * - width  : (integer)
     * - height : (integer)
     */
    protected function transform_fill( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        if (!$this->is_image($params['input_result']['image']))
            { trigger_error(__METHOD__." ERROR: missing input image resource ", E_USER_NOTICE); return false; }
        $image = clone $params['input_result']['image'];

        if (empty($params['transform']['width']) && empty($params['transform']['height']))
            { trigger_error(__METHOD__." ERROR: missing transform param 'width' and/or 'height' ", E_USER_NOTICE); return false; }

        $width  = intval($params['transform']['width'])  ?? 0;
        $height = intval($params['transform']['height']) ?? 0;
        $crop_x = 0;
        $crop_y = 0;

        // transfrom
        if (!empty($width) && !empty($height)) {
            $dest_height = floor(($image->getImageHeight() / $image->getImageWidth()) * $width);
            $dest_width  = floor(($image->getImageWidth() / $image->getImageHeight()) * $height);

            if ($dest_height > $height) {
                $image->resizeImage( $width, 0, Imagick::FILTER_LANCZOS, 1 );
                $crop_y = floor(($dest_height - $height) / 2);  // center

            } else {
                $image->resizeImage( 0, $height, Imagick::FILTER_LANCZOS, 1 );
                $crop_x = floor(($dest_width - $width) / 2);  // center
            }

            $image->cropImage( $width, $height, $crop_x, $crop_y );

        } else {
            $image->resizeImage( $width, $height, Imagick::FILTER_LANCZOS, 1, ($width && $height) );

        }
        $image->setImagePage( $width, $height, 0, 0 );

        $result = array(
            'image'    => $image
            , 'width'  => $image->getImageWidth()
            , 'height' => $image->getImageHeight()
            );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Stretch image (non-proportional) to fill requested width and height
     *
     * @param array $params
     * - REQUIRED
     * - input_result['image']  : (object) input image
     * - REQUIRED - one of the following
     * - transform['width']     : (integer) pixels
     * - transform['height']    : (integer) pixels
     *
     * @return mixed    : false for error, or image info array
     * - image  : (object)
     * - width  : (integer)
     * - height : (integer)
     */
    protected function transform_resize( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        if (!$this->is_image($params['input_result']['image']))
            { trigger_error(__METHOD__." ERROR: missing input image resource ", E_USER_NOTICE); return false; }
        $image = clone $params['input_result']['image'];

        if (empty($params['transform']['width']) && empty($params['transform']['height']))
            { trigger_error(__METHOD__." ERROR: missing transform param 'width' and/or 'height' ", E_USER_NOTICE); return false; }

        $width  = intval($params['transform']['width']  ?? 0);
        $height = intval($params['transform']['height'] ?? 0);

        // transfrom
        $image->resizeImage( $width, $height, Imagick::FILTER_LANCZOS, 1 );
        //$image->setImagePage( $width, $height, 0, 0 );

        $result = array(
            'image'    => $image
            , 'width'  => $image->getImageWidth()
            , 'height' => $image->getImageHeight()
            );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Extract a width x height portion of the image, starting from the top,left position
     *
     * @param array $params
     * - REQUIRED
     * - input_result['image']  : (object) input image
     * - REQUIRED - one of the following
     * - transform['top']       : (integer) pixels
     * - transform['left']      : (integer) pixels
     * - transform['width']     : (integer) pixels
     * - transform['height']    : (integer) pixels
     *
     * @return mixed    : false for error, or image info array
     * - image  : (object)
     * - width  : (integer)
     * - height : (integer)
     */
    protected function transform_crop( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        if (!$this->is_image($params['input_result']['image']))
            { trigger_error(__METHOD__." ERROR: missing input image resource ", E_USER_NOTICE); return false; }
        $image = clone $params['input_result']['image'];

        if (empty($params['transform']['top']) && empty($params['transform']['left']) && empty($params['transform']['width']) && empty($params['transform']['height']))
            { trigger_error(__METHOD__." ERROR: missing transform param 'top', 'left', 'width' or 'height' ", E_USER_NOTICE); return false; }

        $top    = intval($params['transform']['top'])    ?? 0;
        $left   = intval($params['transform']['left'])   ?? 0;
        $width  = intval($params['transform']['width'])  ?? $image->getImageWidth();
        $height = intval($params['transform']['height']) ?? $image->getImageHeight();

        $width  = min($width,  $image->getImageWidth()  - $left);
        $height = min($height, $image->getImageHeight() - $top);
        $x = $left;
        $y = $top;
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('width','height','x','y'),true), E_USER_NOTICE); }

        // transfrom
        $image->cropImage( $width, $height, $x, $y );
        $image->setImagePage( $width, $height, 0, 0 );

        $result = array(
            'image'    => $image
            , 'width'  => $image->getImageWidth()
            , 'height' => $image->getImageHeight()
            );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Resize the image canvas by removing pixels along each edge.
     *
     * @param array $params
     * - REQUIRED
     * - input_result['image']  : (object) input image
     * - REQUIRED - one of the following
     * - transform['top']       : (integer) pixels
     * - transform['left']      : (integer) pixels
     * - transform['bottom']    : (integer) pixels
     * - transform['right']     : (integer) pixels
     *
     * @return mixed    : false for error, or image info array
     * - image  : (object)
     * - width  : (integer)
     * - height : (integer)
     */
    protected function transform_trim( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        if (!$this->is_image($params['input_result']['image']))
            { trigger_error(__METHOD__." ERROR: missing input image resource ", E_USER_NOTICE); return false; }
        $image = clone $params['input_result']['image'];

        if (empty($params['transform']['top']) && empty($params['transform']['left']) && empty($params['transform']['bottom']) && empty($params['transform']['right']))
            { trigger_error(__METHOD__." ERROR: missing transform param 'top', 'left', 'bottom' or 'right' ", E_USER_NOTICE); return false; }

        $top    = intval($params['transform']['top'])    ?? 0;
        $left   = intval($params['transform']['left'])   ?? 0;
        $bottom = intval($params['transform']['bottom']) ?? 0;
        $right  = intval($params['transform']['right'])  ?? 0;

        $width  = $image->getImageWidth()  - $left - $right;
        $height = $image->getImageHeight() - $top  - $bottom;
        $x = $left;
        $y = $top;
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('width','height','x','y'),true), E_USER_NOTICE); }

        // transfrom
        $image->cropImage( $width, $height, $x, $y );
        $image->setImagePage( $width, $height, 0, 0 );

        $result = array(
            'image'    => $image
            , 'width'  => $image->getImageWidth()
            , 'height' => $image->getImageHeight()
            );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Resize the image canvas by adding pixels along each edge.
     *
     * @param array $params
     * - REQUIRED
     * - input_result['image']  : (object) input image
     * - REQUIRED - one of the following
     * - transform['top']       : (integer) pixels
     * - transform['left']      : (integer) pixels
     * - transform['bottom']    : (integer) pixels
     * - transform['right']     : (integer) pixels
     * - OPTIONAL
     * - transform['color_r'] : (integer) bgcolor 0..255
     * - transform['color_g'] : (integer) bgcolor 0..255
     * - transform['color_b'] : (integer) bgcolor 0..255
     * - transform['color_a'] : (float)   opacity 0..1.0
     *
     * @return mixed    : false for error, or image info array
     * - image  : (object)
     * - width  : (integer)
     * - height : (integer)
     */
    protected function transform_border( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        if (!$this->is_image($params['input_result']['image']))
            { trigger_error(__METHOD__." ERROR: missing input image resource ", E_USER_NOTICE); return false; }
        $image = clone $params['input_result']['image'];

        if (empty($params['transform']['top']) && empty($params['transform']['left']) && empty($params['transform']['bottom']) && empty($params['transform']['right']))
            { trigger_error(__METHOD__." ERROR: missing transform param 'top', 'left', 'bottom' or 'right' ", E_USER_NOTICE); return false; }

        $top    = intval($params['transform']['top'])    ?? 0;
        $left   = intval($params['transform']['left'])   ?? 0;
        $bottom = intval($params['transform']['bottom']) ?? 0;
        $right  = intval($params['transform']['right'])  ?? 0;

        $width  = $image->getImageWidth()  - $left - $right;
        $height = $image->getImageHeight() - $top  - $bottom;
        $x = $left;
        $y = $top;
//$border_params = $params['transform'];
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('border_params','width','height','x','y'),true), E_USER_NOTICE); }

        $color_r = abs($params['transform']['color_r'] ?? 0);
        $color_r = min( abs($color_r), 255 );

        $color_g = abs($params['transform']['color_g'] ?? 0);
        $color_g = min( abs($color_g), 255 );

        $color_b = abs($params['transform']['color_b'] ?? 0);
        $color_b = min( abs($color_b), 255 );

        $color_a = floatval($params['transform']['color_a'] ?? 1.0);
        $color_a = min( abs($color_a), 1.0 );
//$border_params = $params['transform'];
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('border_params','top','left','bottom','right'),true), E_USER_NOTICE); }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('color_r','color_g','color_b','color_a'),true), E_USER_NOTICE); }

        // transfrom
        $bgcolor = new ImagickPixel("rgba($color_r, $color_g, $color_b, $color_a)");
        $image->borderImage( $bgcolor, max($top,$bottom), max($left,$right) );
        if ($top != $bottom || $left != $right)
        {
            $crop_top    = ($top < $bottom) ? $bottom - $top : 0;
            $crop_left   = ($left < $right) ? $right - $left : 0;
            $crop_bottom = ($bottom < $top) ? $top - $bottom : 0;
            $crop_right  = ($right < $left) ? $left - $right : 0;
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('crop_top','crop_left','crop_bottom','crop_right'),true), E_USER_NOTICE); }

            $width  = $image->getImageWidth()  - $crop_left - $crop_right;
            $height = $image->getImageHeight() - $crop_top - $crop_bottom;
            $x = $crop_left;
            $y = $crop_top;
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('width','height','x','y'),true), E_USER_NOTICE); }

            $image->cropImage( $width, $height, $x, $y );
        }
        $image->setImagePage( $width, $height, 0, 0 );

        $result = array(
            'image'    => $image
            , 'width'  => $image->getImageWidth()
            , 'height' => $image->getImageHeight()
            );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Flip the image horizontally (x-axis) or vertically (y-axis).
     *
     * @param array $params
     * - REQUIRED
     * - input_result['image']  : (object) input image
     * - transform['axis']      : (string) "x" || "y"
     *
     * @return mixed    : false for error, or image info array
     * - image  : (object)
     * - width  : (integer)
     * - height : (integer)
     */
    protected function transform_flip( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        if (!$this->is_image($params['input_result']['image']))
            { trigger_error(__METHOD__." ERROR: missing input image resource ", E_USER_NOTICE); return false; }
        $image = clone $params['input_result']['image'];

        if (empty($params['transform']['axis']))
            { trigger_error(__METHOD__." ERROR: missing transform param 'axis' ", E_USER_NOTICE); return false; }

        // transfrom
        ($params['transform']['axis'] == 'x') ? $image->flipImage() : $image->flopImage();

        $result = array(
            'image'    => $image
            , 'width'  => $image->getImageWidth()
            , 'height' => $image->getImageHeight()
            );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Rotate the image clockwise.
     *
     * @param array $params
     * - REQUIRED
     * - input_result['image']  : (object) input image
     * - transform['degrees']   : (integer) clockwise degrees from vertical
     * - OPTIONAL
     * - transform['color_r'] : (integer) bgcolor 0..255
     * - transform['color_g'] : (integer) bgcolor 0..255
     * - transform['color_b'] : (integer) bgcolor 0..255
     * - transform['color_a'] : (float)   opacity 0..1.0
     *
     * @return mixed    : false for error, or image info array
     * - image  : (object)
     * - width  : (integer)
     * - height : (integer)
     */
    protected function transform_rotate( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        if (!$this->is_image($params['input_result']['image']))
            { trigger_error(__METHOD__." ERROR: missing input image resource ", E_USER_NOTICE); return false; }
        $image = clone $params['input_result']['image'];

        if (empty($params['transform']['degrees']))
            { trigger_error(__METHOD__." ERROR: missing transform param 'degrees' ", E_USER_NOTICE); return false; }

        $degrees = intval($params['transform']['degrees'] ?? 0);
        $degrees = intval($degrees % 360);

        $color_r = abs($params['transform']['color_r'] ?? 0);
        $color_r = min( abs($color_r), 255 );

        $color_g = abs($params['transform']['color_g'] ?? 0);
        $color_g = min( abs($color_g), 255 );

        $color_b = abs($params['transform']['color_b'] ?? 0);
        $color_b = min( abs($color_b), 255 );

        //$color_a = floatval($params['transform']['color_a'] ?? 1.0);
        $color_a = floatval( (isset($params['transform']['color_a']))
                ? $params['transform']['color_a']
                : (($color_r || $color_g || $color_b) ? 1.0 : 0.0)  // opaque : transparent
                );
        $color_a = min( abs($color_a), 1.0 );

//$rotate_params = $params['transform'];
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('rotate_params','degrees','color_r','color_g','color_b','color_a'),true), E_USER_NOTICE); }

        // transfrom - transparent = '#00000000' || 'rgba(0, 0, 0, 0.0)'
        $bgcolor = new ImagickPixel("rgba($color_r, $color_g, $color_b, $color_a)");
        $image->rotateImage($bgcolor, $degrees);

        $result = array(
            'image'    => $image
            , 'width'  => $image->getImageWidth()
            , 'height' => $image->getImageHeight()
            );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

/*
resizeImage filters:
FILTER_POINT     : 0.334532976151 seconds
FILTER_BOX       : 0.777871131897 seconds
FILTER_TRIANGLE  : 1.3695909977 seconds
FILTER_HERMITE   : 1.35866093636 seconds
FILTER_HANNING   : 4.88722896576 seconds
FILTER_HAMMING   : 4.88665103912 seconds
FILTER_BLACKMAN  : 4.89026689529 seconds
FILTER_GAUSSIAN  : 1.93553304672 seconds
FILTER_QUADRATIC : 1.93322920799 seconds
FILTER_CUBIC     : 2.58396601677 seconds
FILTER_CATROM    : 2.58508896828 seconds     ?
FILTER_MITCHELL  : 2.58368492126 seconds
FILTER_LANCZOS   : 3.74232912064 seconds     *
FILTER_BESSEL    : 4.03305602074 seconds
FILTER_SINC      : 4.90098690987 seconds
*/
    /**
     * Resize image proportionally
     *
     * @param array $params
     * - REQUIRED
     * - input_result['image']  : (object) input image
     * - transform['percent']   : (integer) proportional dimension of input image
     *
     * @return mixed    : false for error, or image info array
     * - image  : (object)
     * - width  : (integer)
     * - height : (integer)
     */
    protected function transform_scale( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        if (!$this->is_image($params['input_result']['image']))
            { trigger_error(__METHOD__." ERROR: missing input image resource ", E_USER_NOTICE); return false; }
        $image = clone $params['input_result']['image'];

        if (empty($params['transform']['percent']))
            { trigger_error(__METHOD__." ERROR: missing transform param 'percent' ", E_USER_NOTICE); return false; }

        $percent = intval($params['transform']['percent'] ?? 100);

        $width   = floor($image->getImageWidth()  * ($percent/100));
        $height  = floor($image->getImageHeight() * ($percent/100));

        // transfrom
        $image->resizeImage( $width, $height, Imagick::FILTER_LANCZOS, 1, ($width && $height) );
        $image->setImagePage( $width, $height, 0, 0 );

        $result = array(
            'image'    => $image
            , 'width'  => $image->getImageWidth()
            , 'height' => $image->getImageHeight()
            );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Resize and crop image proportionally
     *
     * @param array $params
     * - REQUIRED
     * - input_result['image']  : (object) input image
     * - transform['percent']   : (integer) proportional dimension of input image
     * - OPTIONAL
     * - transform['width']  : (integer) default = input width
     * - transform['height'] : (integer) default = input height
     * - transform['focus']  : (integer) default = X, center
     *
     * @return mixed    : false for error, or image info array
     * - image  : (object)
     * - width  : (integer)
     * - height : (integer)
     */
    protected function transform_zoom( $params = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        if (!$this->is_image($params['input_result']['image']))
            { trigger_error(__METHOD__." ERROR: missing input image resource ", E_USER_NOTICE); return false; }
        $image = clone $params['input_result']['image'];

        if (empty($params['transform']['percent']))
            { trigger_error(__METHOD__." ERROR: missing transform param 'percent' ", E_USER_NOTICE); return false; }

        $percent = intval($params['transform']['percent'] ?? 100);
        $focus   = strtolower($params['transform']['focus'] ?? 'x');
        $width   = intval($params['transform']['width']  ?? 0);
        $height  = intval($params['transform']['height'] ?? 0);

        $start_width  = $image->getImageWidth();
        $start_height = $image->getImageHeight();
        $full_width   = floor($start_width  * ($percent/100));
        $full_height  = floor($start_height * ($percent/100));

        // set destination
        if (empty($width) && empty($height)) {
            $width  = $start_width;
            $height = $start_height;
        } else if (!empty($width) && empty($height)) {
            $height = floor(($full_height / $full_width) * $width);
        } else if (empty($width) && !empty($height)) {
            $width  = floor(($full_width / $full_height) * $height);
        }
        $width  = min($width,  $full_width);
        $height = min($height, $full_height);

        // set source
        $src = array(
            'x'   => 0
            , 'y' => 0
            , 'w' => floor(($width  / $full_width)  * $start_width)
            , 'h' => floor(($height / $full_height) * $start_height)
            );

        switch ($focus)
        {
            case 'n':
            case 'north':
                $src['x'] = floor(($start_width  - $src['w']) / 2);
                $src['y'] = 0;
                break;
            case 's':
            case 'south':
                $src['x'] = floor(($start_width  - $src['w']) / 2);
                $src['y'] = $start_height - $src['h'];
                break;
             case 'e':
            case 'east':
                $src['x'] = $start_width  - $src['w'];
                $src['y'] = floor(($start_height - $src['h']) / 2);
                break;
           case 'w':
            case 'west':
                $src['x'] = 0;
                $src['y'] = floor(($start_height - $src['h']) / 2);
                break;

             case 'ne':
            case 'northeast':
                $src['x'] = $start_width  - $src['w'];
                $src['y'] = 0;
                break;
           case 'nw':
            case 'northwest':
                $src['x'] = 0;
                $src['y'] = 0;
                break;
             case 'se':
            case 'southeast':
                $src['x'] = $start_width  - $src['w'];
                $src['y'] = $start_height - $src['h'];
                break;
           case 'sw':
            case 'southwest':
                $src['x'] = 0;
                $src['y'] = $start_height - $src['h'];
                break;

            case 'x':
            case 'center':
            default:
                $src['x'] = floor(($start_width  - $src['w']) / 2);
                $src['y'] = floor(($start_height - $src['h']) / 2);
                break;
        }
// $zoom_params = $params['transform'];
// if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('zoom_params','src','dest'),true), E_USER_NOTICE); }

        // crop source
        $image->cropImage( $src['w'], $src['h'], $src['x'], $src['y'] );
        //$image->setImagePage( $src['w'], $src['h'], 0, 0 );

        // resize
        $image->resizeImage( $width, $height, Imagick::FILTER_LANCZOS, 1, ($width && $height) );
        $image->setImagePage( $width, $height, 0, 0 );

        $result = array(
            'image'    => $image
            , 'width'  => $image->getImageWidth()
            , 'height' => $image->getImageHeight()
            );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('result'),true), E_USER_NOTICE); }
        return $result;
    }


}  // end class IMG_Processor_Imagick
