<?php
/**
 * Shortcodes
 *
 * usage:
 * require_once plugin_dir_path( __FILE__ ) . 'includes/shortcode.php';
 * IMG_Processor_Shortcode::init();
 *
 * @package           tenandtwo-wp-plugins
 * @subpackage        tenandtwo-img-processor
 * @author            Ten & Two Systems
 * @copyright         2023 Ten & Two Systems
 */
defined( 'ABSPATH' ) or die( 'Not for browsing' );


class IMG_Processor_Shortcode
{

    /**
     * add_shortcodes   : img_process, img_info, img_scale, etc
     */
    public static function init()
    {
//if (WP_DEBUG) { trigger_error(__METHOD__, E_USER_NOTICE); }

        $options = get_option( IMG_PROCESSOR_OPTS, array() );
        if (empty($options['sc_img_process'])) { return; }

        add_filter( 'no_texturize_shortcodes',  array('IMG_Processor_Shortcode', 'no_texturize_shortcodes') );

        if (!shortcode_exists('img_process'))
            { add_shortcode( 'img_process', array('IMG_Processor_Shortcode', 'img_process') ); }
        elseif (WP_DEBUG)
            { trigger_error(__METHOD__." : shortcode 'img_process' already exists", E_USER_NOTICE); }

        add_shortcode( 'img_info',      array('IMG_Processor_Shortcode', 'img_info') );
        add_shortcode( 'img_fit',       array('IMG_Processor_Shortcode', 'img_fit') );
        add_shortcode( 'img_fill',      array('IMG_Processor_Shortcode', 'img_fill') );
        add_shortcode( 'img_resize',    array('IMG_Processor_Shortcode', 'img_resize') );
        add_shortcode( 'img_crop',      array('IMG_Processor_Shortcode', 'img_crop') );
        add_shortcode( 'img_trim',      array('IMG_Processor_Shortcode', 'img_trim') );
        add_shortcode( 'img_border',    array('IMG_Processor_Shortcode', 'img_border') );
        add_shortcode( 'img_flip',      array('IMG_Processor_Shortcode', 'img_flip') );
        add_shortcode( 'img_rotate',    array('IMG_Processor_Shortcode', 'img_rotate') );
        add_shortcode( 'img_scale',     array('IMG_Processor_Shortcode', 'img_scale') );
        add_shortcode( 'img_zoom',      array('IMG_Processor_Shortcode', 'img_zoom') );

        if (IMG_PROCESSOR_IMAGICK && false)
        {
//             add_shortcode( 'img_colorize',  array('IMG_Processor_Shortcode', 'img_colorize') );
//             add_shortcode( 'img_grayscale', array('IMG_Processor_Shortcode', 'img_grayscale') );
//             add_shortcode( 'img_negate',    array('IMG_Processor_Shortcode', 'img_negate') );
        }

        if (WP_DEBUG && !shortcode_exists('img_test'))
            { add_shortcode( 'img_test', array('IMG_Processor_Shortcode', 'img_test') ); }

    }


    /**
     * content filter : no_texturize_shortcodes
     */
    public static function no_texturize_shortcodes( $shortcodes )
    {
        if (shortcode_exists('img_process'))    { $shortcodes[] = 'img_process'; }
        if (shortcode_exists('img_info'))       { $shortcodes[] = 'img_info'; }
        if (shortcode_exists('img_fit'))        { $shortcodes[] = 'img_fit'; }
        if (shortcode_exists('img_fill'))       { $shortcodes[] = 'img_fill'; }
        if (shortcode_exists('img_resize'))     { $shortcodes[] = 'img_resize'; }
        if (shortcode_exists('img_crop'))       { $shortcodes[] = 'img_crop'; }
        if (shortcode_exists('img_trim'))       { $shortcodes[] = 'img_trim'; }
        if (shortcode_exists('img_border'))     { $shortcodes[] = 'img_border'; }
        if (shortcode_exists('img_flip'))       { $shortcodes[] = 'img_flip'; }
        if (shortcode_exists('img_rotate'))     { $shortcodes[] = 'img_rotate'; }
        if (shortcode_exists('img_scale'))      { $shortcodes[] = 'img_scale'; }
        if (shortcode_exists('img_zoom'))       { $shortcodes[] = 'img_zoom'; }

//         if (shortcode_exists('img_colorize'))   { $shortcodes[] = 'img_colorize'; }
//         if (shortcode_exists('img_grayscale'))  { $shortcodes[] = 'img_grayscale'; }
//         if (shortcode_exists('img_negate'))     { $shortcodes[] = 'img_negate'; }

        if (shortcode_exists('img_test'))   { $shortcodes[] = 'img_test'; }
        return $shortcodes;
    }


    /**
     * shortcode : img_test
     */
    public static function img_test( $attrs, $content )
    {
if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs','content'),true), E_USER_NOTICE); }
        $result = "this is a test";
        return print_r($result,true);
    }


    /**
     * shortcode : img_process
     * @param content               : file|url
     * @param attribute 'source'    : file|url
     * @param attribute 'output'    : string, info|array_keys(info)|json|img|html|figure
     *
     * @param attribute 'transform' : string, none|fit|fill|resize|crop|trim|border|flip|rotate|scale|zoom
     *
     * @returns string from img_result()
     */
    public static function img_process( $attrs, $content )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs','content'),true), E_USER_NOTICE); }

        $attrs = array_change_key_case( (array)$attrs, CASE_LOWER );
        if (!isset($attrs['transform']))
            { return wp_kses("Shortcode Error : '".__FUNCTION__."' missing input attribute 'transform' ", 'post'); }

        $transform = $attrs['transform'] ?? 'none';  // or 'info'

        if ($transform == 'fit')    { return self::img_fit(     $attrs, $content ); }
        if ($transform == 'fill')   { return self::img_fill(    $attrs, $content ); }
        if ($transform == 'resize') { return self::img_resize(  $attrs, $content ); }
        if ($transform == 'crop')   { return self::img_crop(    $attrs, $content ); }
        if ($transform == 'trim')   { return self::img_trim(    $attrs, $content ); }
        if ($transform == 'border') { return self::img_border(  $attrs, $content ); }
        if ($transform == 'flip')   { return self::img_flip(    $attrs, $content ); }
        if ($transform == 'rotate') { return self::img_rotate(  $attrs, $content ); }
        if ($transform == 'scale')  { return self::img_scale(   $attrs, $content ); }
        if ($transform == 'zoom')   { return self::img_zoom(    $attrs, $content ); }

//         if ($transform == 'colorize')    { return self::img_colorize(  $attrs, $content ); }
//         if ($transform == 'grayscale')   { return self::img_grayscale( $attrs, $content ); }
//         if ($transform == 'negate')      { return self::img_negate(    $attrs, $content ); }

        return self::img_info( $attrs, $content );
    }


    /**
     * shortcode : img_info
     * @param content               : file|url
     * @param attribute 'source'    : file|url
     * @param attribute 'output'    : string, info|array_keys(info)|json|img|html|figure
     *
     * @param none
     *
     * @returns string from img_result()
     */
    public static function img_info( $attrs, $content )
    {
        $attrs = array_change_key_case( (array)$attrs, CASE_LOWER );
        $attrs['content'] = $content;
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs'),true), E_USER_NOTICE); }

        if (empty($attrs['output'])) { $attrs['output'] = 'info'; }  // override default

        $transform = array(
            'type' => 'none'
            );
        $image = self::img_cache( $attrs, $transform );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs','transform','image'),true), E_USER_NOTICE); }
        return (!empty($image['url']))
            ? self::img_result( $attrs, $image )
            : wp_kses("Shortcode Error : '".__FUNCTION__."' ".print_r($image,true), 'post');
    }


    /**
     * shortcode : img_fit
     * @param content               : file|url
     * @param attribute 'source'    : file|url
     * @param attribute 'output'    : string, info|array_keys(info)|json|img|html|figure
     *
     * @param attribute 'width'     : integer, max pixels
     * @param attribute 'height'    : integer, max pixels
     *
     * @returns string from img_result()
     */
    public static function img_fit( $attrs, $content )
    {
        $attrs = array_change_key_case( (array)$attrs, CASE_LOWER );
        $attrs['content'] = $content;
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs'),true), E_USER_NOTICE); }

        $width  = (isset($attrs['width']))  ? intval($attrs['width'])  : 0;
        $height = (isset($attrs['height'])) ? intval($attrs['height']) : 0;
        if (empty($width) && empty($height))
        {
            $options = get_option( IMG_PROCESSOR_OPTS, array() );
            $width  = $options['width']  ?? 0;
            $height = $options['height'] ?? 0;
        }

        $transform = array(
            'type' => 'fit'
            , 'width'  => $width
            , 'height' => $height
            );
        $image = self::img_cache( $attrs, $transform );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs','transform','image'),true), E_USER_NOTICE); }
        return (!empty($image['url']))
            ? self::img_result( $attrs, $image )
            : wp_kses("Shortcode Error : '".__FUNCTION__."' ".print_r($image,true), 'post');
    }


    /**
     * shortcode : img_fill
     * @param content               : file|url
     * @param attribute 'source'    : file|url
     * @param attribute 'output'    : string, info|array_keys(info)|json|img|html|figure
     *
     * @param attribute 'width'     : integer, max pixels
     * @param attribute 'height'    : integer, max pixels
     *
     * @returns string from img_result()
     */
    public static function img_fill( $attrs, $content )
    {
        $attrs = array_change_key_case( (array)$attrs, CASE_LOWER );
        $attrs['content'] = $content;
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs'),true), E_USER_NOTICE); }

        $width  = (isset($attrs['width']))  ? intval($attrs['width'])  : 0;
        $height = (isset($attrs['height'])) ? intval($attrs['height']) : 0;
        if (empty($width) && empty($height))
        {
            $options = get_option( IMG_PROCESSOR_OPTS, array() );
            $width  = $options['width']  ?? 0;
            $height = $options['height'] ?? 0;
        }

        $transform = array(
            'type' => 'fill'
            , 'width'  => $width
            , 'height' => $height
            );
        $image = self::img_cache( $attrs, $transform );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs','transform','image'),true), E_USER_NOTICE); }
        return (!empty($image['url']))
            ? self::img_result( $attrs, $image )
            : wp_kses("Shortcode Error : '".__FUNCTION__."' ".print_r($image,true), 'post');
    }


    /**
     * shortcode : img_resize
     * @param content               : file|url
     * @param attribute 'source'    : file|url
     * @param attribute 'output'    : string, info|array_keys(info)|json|img|html|figure
     *
     * @param attribute 'width'     : integer, max pixels
     * @param attribute 'height'    : integer, max pixels
     *
     * @returns string from img_result()
     */
    public static function img_resize( $attrs, $content )
    {
        $attrs = array_change_key_case( (array)$attrs, CASE_LOWER );
        $attrs['content'] = $content;
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs'),true), E_USER_NOTICE); }

        $width  = (isset($attrs['width']))  ? intval($attrs['width'])  : 0;
        $height = (isset($attrs['height'])) ? intval($attrs['height']) : 0;
        if (empty($width) && empty($height))
        {
            $options = get_option( IMG_PROCESSOR_OPTS, array() );
            $width  = $options['width']  ?? 0;
            $height = $options['height'] ?? 0;
        }

        $transform = array(
            'type' => 'resize'
            , 'width'  => $width
            , 'height' => $height
            );
        $image = self::img_cache( $attrs, $transform );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs','transform','image'),true), E_USER_NOTICE); }
        return (!empty($image['url']))
            ? self::img_result( $attrs, $image )
            : wp_kses("Shortcode Error : '".__FUNCTION__."' ".print_r($image,true), 'post');
    }


    /**
     * shortcode : img_crop
     * @param content               : file|url
     * @param attribute 'source'    : file|url
     * @param attribute 'output'    : string, info|array_keys(info)|json|img|html|figure
     *
     * @param attribute 'top'       : integer, starting position
     * @param attribute 'left'      : integer, starting position
     * @param attribute 'width'     : integer, max pixels
     * @param attribute 'height'    : integer, max pixels
     *
     * @returns string from img_result()
     */
    public static function img_crop( $attrs, $content )
    {
        $attrs = array_change_key_case( (array)$attrs, CASE_LOWER );
        $attrs['content'] = $content;
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs'),true), E_USER_NOTICE); }

        $top    = (isset($attrs['top']))    ? intval($attrs['top'])    : 0;
        $left   = (isset($attrs['left']))   ? intval($attrs['left'])   : 0;
        $width  = (isset($attrs['width']))  ? intval($attrs['width'])  : 0;
        $height = (isset($attrs['height'])) ? intval($attrs['height']) : 0;
        if (empty($top) && empty($left) && empty($width) && empty($height))
        {
            $options = get_option( IMG_PROCESSOR_OPTS, array() );
            $top    = $options['top']    ?? 0;
            $left   = $options['left']   ?? 0;
            $width  = $options['width']  ?? 0;
            $height = $options['height'] ?? 0;
        }

        $transform = array(
            'type' => 'crop'
            , 'top'    => $top
            , 'left'   => $left
            , 'width'  => $width
            , 'height' => $height
            );
        $image = self::img_cache( $attrs, $transform );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs','transform','image'),true), E_USER_NOTICE); }
        return (!empty($image['url']))
            ? self::img_result( $attrs, $image )
            : wp_kses("Shortcode Error : '".__FUNCTION__."' ".print_r($image,true), 'post');
    }


    /**
     * shortcode : img_trim
     * @param content               : file|url
     * @param attribute 'source'    : file|url
     * @param attribute 'output'    : string, info|array_keys(info)|json|img|html|figure
     *
     * @param attribute 'top'       : integer, pixels to remove
     * @param attribute 'left'      : integer, pixels to remove
     * @param attribute 'bottom'    : integer, pixels to remove
     * @param attribute 'right'     : integer, pixels to remove
     *
     * @returns string from img_result()
     */
    public static function img_trim( $attrs, $content )
    {
        $attrs = array_change_key_case( (array)$attrs, CASE_LOWER );
        $attrs['content'] = $content;
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs'),true), E_USER_NOTICE); }

        $top    = (isset($attrs['top']))    ? intval($attrs['top'])    : 0;
        $left   = (isset($attrs['left']))   ? intval($attrs['left'])   : 0;
        $bottom = (isset($attrs['bottom'])) ? intval($attrs['bottom']) : 0;
        $right  = (isset($attrs['right']))  ? intval($attrs['right'])  : 0;
        if (empty($top) && empty($left) && empty($bottom) && empty($right))
        {
            $options = get_option( IMG_PROCESSOR_OPTS, array() );
            $top    = $options['top']    ?? 0;
            $left   = $options['left']   ?? 0;
            $bottom = $options['bottom'] ?? 0;
            $right  = $options['right']  ?? 0;
        }

        $transform = array(
            'type' => 'trim'
            , 'top'    => $top
            , 'left'   => $left
            , 'bottom' => $bottom
            , 'right'  => $right
            );
        $image = self::img_cache( $attrs, $transform );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs','transform','image'),true), E_USER_NOTICE); }
        return (!empty($image['url']))
            ? self::img_result( $attrs, $image )
            : wp_kses("Shortcode Error : '".__FUNCTION__."' ".print_r($image,true), 'post');
    }


    /**
     * shortcode : img_border
     * @param content               : file|url
     * @param attribute 'source'    : file|url
     * @param attribute 'output'    : string, info|array_keys(info)|json|img|html|figure
     *
     * @param attribute 'top'       : integer, pixels to add
     * @param attribute 'left'      : integer, pixels to add
     * @param attribute 'bottom'    : integer, pixels to add
     * @param attribute 'right'     : integer, pixels to add
     * @param attribute 'color_r'   : integer, 0-255 bgcolor
     * @param attribute 'color_g'   : integer, 0-255 bgcolor
     * @param attribute 'color_b'   : integer, 0-255 bgcolor
     * @param attribute 'color_a'   : float,   0.0-1.0 opacity
     *
     * @returns string from img_result()
     */
    public static function img_border( $attrs, $content )
    {
        $attrs = array_change_key_case( (array)$attrs, CASE_LOWER );
        $attrs['content'] = $content;
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs'),true), E_USER_NOTICE); }

        $top    = (isset($attrs['top']))    ? intval($attrs['top'])    : 0;
        $left   = (isset($attrs['left']))   ? intval($attrs['left'])   : 0;
        $bottom = (isset($attrs['bottom'])) ? intval($attrs['bottom']) : 0;
        $right  = (isset($attrs['right']))  ? intval($attrs['right'])  : 0;
        if (empty($top) && empty($left) && empty($bottom) && empty($right))
        {
            $options = get_option( IMG_PROCESSOR_OPTS, array() );
            $top    = $options['top']    ?? 0;
            $left   = $options['left']   ?? 0;
            $bottom = $options['bottom'] ?? 0;
            $right  = $options['right']  ?? 0;
        }

        $transform = array(
            'type' => 'border'
            , 'top'    => $top
            , 'left'   => $left
            , 'bottom' => $bottom
            , 'right'  => $right
            );
        // color params
        if (isset($attrs['color_r'])) { $transform['color_r'] = intval($attrs['color_r']); }
        if (isset($attrs['color_g'])) { $transform['color_g'] = intval($attrs['color_g']); }
        if (isset($attrs['color_b'])) { $transform['color_b'] = intval($attrs['color_b']); }
        if (isset($attrs['color_a'])) { $transform['color_a'] = floatval($attrs['color_a']); }

        $image = self::img_cache( $attrs, $transform );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs','transform','image'),true), E_USER_NOTICE); }
        return (!empty($image['url']))
            ? self::img_result( $attrs, $image )
            : wp_kses("Shortcode Error : '".__FUNCTION__."' ".print_r($image,true), 'post');
    }


    /**
     * shortcode : img_flip
     * @param content               : file|url
     * @param attribute 'source'    : file|url
     * @param attribute 'output'    : string, info|array_keys(info)|json|img|html|figure
     *
     * @param attribute 'axis'      : string, x|y
     *
     * @returns string from img_result()
     */
    public static function img_flip( $attrs, $content )
    {
        $attrs = array_change_key_case( (array)$attrs, CASE_LOWER );
        $attrs['content'] = $content;
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs'),true), E_USER_NOTICE); }

        $axis = strtolower($attrs['axis'] ?? '');
        if (!in_array($axis,array('x','y')))
        {
            $options = get_option( IMG_PROCESSOR_OPTS, array() );
            $axis = $options['axis'] ?? 'y';
        }

        $transform = array(
            'type' => 'flip'
            , 'axis' => $axis
            );
        $image = self::img_cache( $attrs, $transform );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs','transform','image'),true), E_USER_NOTICE); }
        return (!empty($image['url']))
            ? self::img_result( $attrs, $image )
            : wp_kses("Shortcode Error : '".__FUNCTION__."' ".print_r($image,true), 'post');
    }


    /**
     * shortcode : img_rotate
     * @param content               : file|url
     * @param attribute 'source'    : file|url
     * @param attribute 'output'    : string, info|array_keys(info)|json|img|html|figure
     *
     * @param attribute 'degrees'   : integer
     * @param attribute 'color_r'   : integer, 0-255 background color
     * @param attribute 'color_g'   : integer, 0-255 background color
     * @param attribute 'color_b'   : integer, 0-255 background color
     * @param attribute 'color_a'   : float,   0.0-1.0 opacity
     *
     * @returns string from img_result()
     */
    public static function img_rotate( $attrs, $content )
    {
        $attrs = array_change_key_case( (array)$attrs, CASE_LOWER );
        $attrs['content'] = $content;
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs'),true), E_USER_NOTICE); }

        $degrees = (isset($attrs['degrees'])) ? intval($attrs['degrees']) : 0;
        if (empty($degrees))
        {
            $options = get_option( IMG_PROCESSOR_OPTS, array() );
            $degrees = $options['degrees'] ?? 90;
        }

        $transform = array(
            'type' => 'rotate'
            , 'degrees' => $degrees
            );
        // color params
        if (isset($attrs['color_r'])) { $transform['color_r'] = intval($attrs['color_r']); }
        if (isset($attrs['color_g'])) { $transform['color_g'] = intval($attrs['color_g']); }
        if (isset($attrs['color_b'])) { $transform['color_b'] = intval($attrs['color_b']); }
        if (isset($attrs['color_a'])) { $transform['color_a'] = floatval($attrs['color_a']); }

        $image = self::img_cache( $attrs, $transform );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs','transform','image'),true), E_USER_NOTICE); }
        return (!empty($image['url']))
            ? self::img_result( $attrs, $image )
            : wp_kses("Shortcode Error : '".__FUNCTION__."' ".print_r($image,true), 'post');
    }


    /**
     * shortcode : img_scale
     * @param content               : file|url
     * @param attribute 'source'    : file|url
     * @param attribute 'output'    : string, info|array_keys(info)|json|img|html|figure
     *
     * @param attribute 'percent'   : integer
     *
     * @returns string from img_result()
     */
    public static function img_scale( $attrs, $content )
    {
        $attrs = array_change_key_case( (array)$attrs, CASE_LOWER );
        $attrs['content'] = $content;
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs'),true), E_USER_NOTICE); }

        $percent = (isset($attrs['percent'])) ? intval($attrs['percent']) : 0;
        if (empty($percent))
        {
            $options = get_option( IMG_PROCESSOR_OPTS, array() );
            $percent = $options['percent'] ?? 100;
        }

        $transform = array(
            'type' => 'scale'
            , 'percent' => $percent
            );
        $image = self::img_cache( $attrs, $transform );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs','transform','image'),true), E_USER_NOTICE); }
        return (!empty($image['url']))
            ? self::img_result( $attrs, $image )
            : wp_kses("Shortcode Error : '".__FUNCTION__."' ".print_r($image,true), 'post');
    }


    /**
     * shortcode : img_zoom
     * @param content               : file|url
     * @param attribute 'source'    : file|url
     * @param attribute 'output'    : string, info|array_keys(info)|json|img|html|figure
     *
     * @param attribute 'percent'   : integer
     * @param attribute 'focus'     : string, X|N|S|E|W|NE|NW|SE|SW
     * @param attribute 'width'     : integer, pixels
     * @param attribute 'height'    : integer, pixels
     *
     * @returns string from img_result()
     */
    public static function img_zoom( $attrs, $content )
    {
        $attrs = array_change_key_case( (array)$attrs, CASE_LOWER );
        $attrs['content'] = $content;
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs'),true), E_USER_NOTICE); }

        $percent = (isset($attrs['percent'])) ? intval($attrs['percent']) : 0;
        $focus   = (isset($attrs['focus']))   ? strval($attrs['focus'])   : null;
        $width   = (isset($attrs['width']))   ? intval($attrs['width'])   : 0;
        $height  = (isset($attrs['height']))  ? intval($attrs['height'])  : 0;
        if (empty($percent))
        {
            $options = get_option( IMG_PROCESSOR_OPTS, array() );
            $percent = $options['percent'] ?? 100;
        }

        $transform = array(
            'type' => 'zoom'
            , 'percent' => $percent
            , 'focus'   => $focus
            , 'width'   => $width
            , 'height'  => $height
            );
        $image = self::img_cache( $attrs, $transform );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs','transform','image'),true), E_USER_NOTICE); }
        return (!empty($image['url']))
            ? self::img_result( $attrs, $image )
            : wp_kses("Shortcode Error : '".__FUNCTION__."' ".print_r($image,true), 'post');
    }


    /**
     * shortcode : img_colorize
     * @param content               : file|url
     * @param attribute 'source'    : file|url
     * @param attribute 'output'    : string, info|array_keys(info)|json|img|html|figure
     *
     * @param attribute 'color_r'   : integer, -255..0..+255
     * @param attribute 'color_g'   : integer, -255..0..+255
     * @param attribute 'color_b'   : integer, -255..0..+255
     * @param attribute 'color_a'   : float,   0.0..1.0
     *
     * @returns string from img_result()
     */
    public static function img_colorize( $attrs, $content )
    {
        $attrs = array_change_key_case( (array)$attrs, CASE_LOWER );
        $attrs['content'] = $content;
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs'),true), E_USER_NOTICE); }

        $transform = array(
            'type' => 'colorize'
            );
        if (isset($attrs['color_r'])) { $transform['color_r'] = intval($attrs['color_r']); }
        if (isset($attrs['color_g'])) { $transform['color_g'] = intval($attrs['color_g']); }
        if (isset($attrs['color_b'])) { $transform['color_b'] = intval($attrs['color_b']); }
        if (isset($attrs['color_a'])) { $transform['color_a'] = floatval($attrs['color_a']); }

        $image = self::img_cache( $attrs, $transform );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs','transform','image'),true), E_USER_NOTICE); }
        return (!empty($image['url']))
            ? self::img_result( $attrs, $image )
            : wp_kses("Shortcode Error : '".__FUNCTION__."' ".print_r($image,true), 'post');
    }


    /**
     * shortcode : img_grayscale
     * @param content               : file|url
     * @param attribute 'source'    : file|url
     * @param attribute 'output'    : string, info|array_keys(info)|json|img|html|figure
     *
     * @param none
     *
     * @returns string from img_result()
     */
    public static function img_grayscale( $attrs, $content )
    {
        $attrs = array_change_key_case( (array)$attrs, CASE_LOWER );
        $attrs['content'] = $content;
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs'),true), E_USER_NOTICE); }

        $transform = array(
            'type'  => 'grayscale'
            );
        $image = self::img_cache( $attrs, $transform );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs','transform','image'),true), E_USER_NOTICE); }
        return (!empty($image['url']))
            ? self::img_result( $attrs, $image )
            : wp_kses("Shortcode Error : '".__FUNCTION__."' ".print_r($image,true), 'post');
    }


    /**
     * shortcode : img_negate
     * @param content               : file|url
     * @param attribute 'source'    : file|url
     * @param attribute 'output'    : string, info|array_keys(info)|json|img|html|figure
     *
     * @param none
     *
     * @returns string from img_result()
     */
    public static function img_negate( $attrs, $content )
    {
        $attrs = array_change_key_case( (array)$attrs, CASE_LOWER );
        $attrs['content'] = $content;
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs'),true), E_USER_NOTICE); }

        $transform = array(
            'type'  => 'negate'
            );
        $image = self::img_cache( $attrs, $transform );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs','transform','image'),true), E_USER_NOTICE); }
        return (!empty($image['url']))
            ? self::img_result( $attrs, $image )
            : wp_kses("Shortcode Error : '".__FUNCTION__."' ".print_r($image,true), 'post');
    }


    /**
     * @param attrs     : array, shortcode attributes
     * @param transform : array, getImageCache() params
     *
     * @returns array   : cached image file info
     */
    protected static function img_cache( $attrs, $transform )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs','transform'),true), E_USER_NOTICE); }

        $options = get_option( IMG_PROCESSOR_OPTS, array() );
        $attrs = array_change_key_case( (array)$attrs, CASE_LOWER );

//wp_get_attachment_image_url( int $attachment_id, string|int[] $size = ‘thumbnail’, bool $icon = false ): string|false

        $src = $attrs['source'] ?? null;
        if (!empty($attrs['content']))
            { $src = do_shortcode( $attrs['content'] ) ?: $src; }
        if (empty($src))
            { return sprintf( esc_html__( "missing input attribute 'source' ", 'tenandtwo-img-processor' ) ); }

        $path = null;
        if (filter_var($src, FILTER_VALIDATE_URL)) {
            $path = IMG_Processor_Util::getFileExistsRemote( $src );
        } else {
            $search_paths = explode("\n", $options['search_path'] ?? '');
            $search_paths[] = ABSPATH;
            $path = IMG_Processor_Util::getFileExistsLocal( $src, $search_paths );
        }
        if (empty($path))
           { $path = $options['placeholder'] ?? IMG_PROCESSOR_PLACEHOLDER; }

        $params = array(
            'input' => array(
                'file' => $path
                ),
            'transform' => $transform,
            'output' => array(
                'file'   => $attrs['output_file'] ?? null
                , 'type' => $attrs['output_type'] ?? null
                , 'dpi'  => $attrs['output_dpi']  ?? null
                ),
            );
        if (IMG_PROCESSOR_IMAGICK)
        {
            global $IMG_MGR;
            if (empty($IMG_MGR)) { $IMG_MGR = new IMG_Processor_Imagick(); }
            $result = $IMG_MGR->getImageCache( $params );
        }
        else
        {
            global $GD_MGR;
            if (empty($GD_MGR)) { $GD_MGR = new IMG_Processor_GD(); }
            $result = $GD_MGR->getImageCache( $params );
        }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs','params','result'),true), E_USER_NOTICE); }
        return $result;
    }


    /**
     * @param array 'image'  : cached image file info
     *
     * @param attribute 'output'  : 'img' (dflt) || 'info' || array_keys(info) || 'json' || 'html' || 'figure'
     *
     * if output == 'figure'
     * @param attribute 'figure-id'     : add 'id' attribute
     * @param attribute 'figure-class'  : override figure 'class' attribute
     * @param attribute 'caption'       : add <figcaption/> node
     * @param attribute 'caption-class' : override figcaption 'class' attribute
     *
     * if output == 'html' || output == 'figure'
     * @param attribute 'img-id'    : add 'id' attribute
     * @param attribute 'alt'       : override 'alt' attribute
     * @param attribute 'title'     : add 'title' attribute
     * @param attribute 'style'     : add 'style' attribute
     * @param attribute 'decoding'  : override 'decoding' attribute [auto, *async, sync]
     * @param attribute 'loading'   : add 'loading' attribute [eager, lazy]
     *
     * @param attribute 'htmlentities'  : escape return value
     *
     * @returns string
     */
    protected static function img_result( $attrs, $image )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs','image'),true), E_USER_NOTICE); }
        if (!is_array($image))  { return false; }

        //$attrs = array_change_key_case( (array)$attrs, CASE_LOWER );
        $attrs_bool = IMG_Processor_Util::getShortcodeBooleans( array(
            'htmlentities' => false,
        ), $attrs, 'img_result' );

//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs','image'),true), E_USER_NOTICE); }

        $rv = $image['img'] ?? '';  // default
        $output = $attrs['output'] ?? 'img';

        $img_id        = $attrs['img-id']       ?? '';
        $img_alt       = $attrs['alt']          ?? $image['alt'];
        $img_title     = $attrs['title']        ?? '';
        $img_style     = $attrs['style']        ?? '';
        $img_decoding  = $attrs['decoding']     ?? 'async'; // auto, async, sync
        $img_loading   = $attrs['loading']      ?? '';      // eager, lazy

        $figure_id     = $attrs['figure-id']     ?? '';
        $figure_class  = $attrs['figure-class']  ?? 'wp-block-image aligncenter';
        $caption       = $attrs['caption']       ?? '';
        $caption_class = $attrs['caption-class'] ?? 'wp-element-caption';

        // any key from getImageFileInfo()
        if (in_array($output, array_keys($image)))
        {
            $rv = $image[$output] ?? '';
        }
        // <ul/>
        elseif ($output == 'info')
        {
            $rv = '<div class="img-proc-info"><ul>';
//             foreach($image as $key => $val)
//                 { $rv .= "<li><strong>$key:</strong> ".print_r($val,true)."</li>"; }

            foreach($image as $key => $val) {
                $rv .= "<li><strong>$key:</strong> ";
                if (is_array($val)) {
                    $rv .= '<ul>';
                    foreach($val as $k => $v) {
                        $rv .= "<li><strong>$k:</strong> <code>".htmlentities($v)."</code></li>";
                    }
                    $rv .= '</ul>';
                } else {
                    $rv .= print_r($val,true);
                }
                $rv .= "</li>";
            }

            $rv .= '</ul></div>';
        }
        // {json}
        elseif ($output == 'json')
        {
            unset($image['img'],$image['_usage']);
            $rv = json_encode($image);
        }
        elseif ($output == 'html' || $output == 'figure')
        {
        // <img/>
            $rv = '<img';
            if (!empty($img_id))            { $rv .= ' id="'       . esc_attr($img_id) . '"'; }
            if (!empty($img_alt))           { $rv .= ' alt="'      . esc_attr($img_alt) . '"'; }
            if (!empty($img_title))         { $rv .= ' title="'    . esc_attr($img_title) . '"'; }
            if (!empty($img_style))         { $rv .= ' style="'    . esc_attr($img_style) . '"'; }
            if (!empty($img_decoding))      { $rv .= ' decoding="' . esc_attr($img_decoding) . '"'; }
            if (!empty($img_loading))       { $rv .= ' loading="'  . esc_attr($img_loading) . '"'; }
            if (!empty($image['url']))      { $rv .= ' src="'      . esc_attr($image['url']) . '"'; }
            if (!empty($image['width']))    { $rv .= ' width="'    . esc_attr($image['width']) . '"'; }
            if (!empty($image['height']))   { $rv .= ' height="'   . esc_attr($image['height']) . '"'; }
            $rv .= '/>';

        // <figure><img/></figure>
            if ($output == 'figure')
            {
                $fig = '<figure';
                if (!empty($figure_id))     { $fig .= ' id="'    . esc_attr($figure_id) . '"'; }
                if (!empty($figure_class))  { $fig .= ' class="' . esc_attr($figure_class) . '"'; }
                $fig .= '>';

                $fig .= $rv;

                if (!empty($caption)) {
                    $fig .= '<figcaption';
                    if (!empty($caption_class)) { $fig .= ' class="' . esc_attr($caption_class) . '"'; }
                    $fig .= '/>' . esc_html($caption) . '</figcaption>';
                }
                $fig .= '</figure>';

                $rv = $fig;
            }
        }

//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attrs','image','rv'),true), E_USER_NOTICE); }
        return ($attrs_bool['htmlentities']) ? esc_html( $rv ) : wp_kses($rv, 'post');
    }


}  // end IMG_Processor_Shortcode


/* -------------------------
    <figure
        class="wp-block-image size-large has-custom-border newclass is-style-default"
        id="anchor"
        >
    <img
        decoding="async"
        src="/wp-content/plugins/tenandtwo-img-processor/cache/e905436a/_missing_jpeg/scale_50.jpeg"
        alt="alt txt"
        style="border-width:1px;border-radius:6px;aspect-ratio:3/4;object-fit:cover"
        title="tattri"
        />
    </figure>
-------------------------
    figure class=
        wp-block-image
        size-large (840px) | size-medium (512px) | size-small (384px)
        is-style-default | is-style-rounded | is-style-squared | is-style-outline | is-style-circle-mask
        has-custom-border
        alignfull | alignwide | alignleft | aligncenter | alignright
-------------------------
    img style=
        border-width:0px;
        border-radius:0px;
        aspect-ratio:1;
        object-fit:contain

        border-width    : '1px'
        border-radius   : '2px'
        padding         : '3px'
        margin          : '4px'

        aspect-ratio    : '1', '4/3', '3/4', '3/2', '2/3', '16/9', '9/16'
        object-fit      : 'fill', 'contain', 'cover', 'none', 'scale-down'
        object-position : '50% 50%', 'right top', 'left bottom', '10px 20px'

------------------------- */
