<?php
/**
 * Util
 *
 * @package           tenandtwo-wp-plugins
 * @subpackage        tenandtwo-img-processor
 * @author            Ten & Two Systems
 * @copyright         2023 Ten & Two Systems
 */

defined( 'ABSPATH' ) or die( 'Not for browsing' );

//apply_filters( ‘wp_get_attachment_image_attributes’, string[] $attr, WP_Post $attachment, string|int[] $size )
add_action( 'wp_get_attachment_image_attributes', array('IMG_Processor_Util', 'test'), 10, 3 );

class IMG_Processor_Util
{

   public static function test( $attr, $attachment, $size ) {
if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('attr','attachment','size'),true), E_USER_NOTICE); }
    	return $attr;
    }


    /**
     * debug util for timing
     *
     * <code>
     * $starttime = IMG_Processor_Util::getMicrotime();
     * // ...timed code...
     * $stoptime = IMG_Processor_Util::getMicrotime();
     * $runtime = sprintf("%.4f",($stoptime - $starttime)) . " seconds";
     * </code>
     *
     * @see date.xsl, template name="date-microtime"
     * @param none
     * @return float    : current microtime with milliseconds
     */
    public static function getMicrotime()
    {
        list($usec, $sec) = explode(" ",microtime());
        return ((float)$usec + (float)$sec);
    }


    /**
     * @see ???
     * @param array $params
     * - method : string, dflt=md5, sha256, sha384, sha512, ...
     * - data   : string
     * - raw_output : bool
     * @return string
     */
    public static function getHash( $params )
    {
        if (empty($params['method'])) { $params['method'] = "md5"; }
        if (empty($params['data']))   { $params['data'] = ''; }
        $params['raw_output'] = (!empty($params['raw_output']) && $params['raw_output'] != 'false');

        $rv = hash( $params['method'], $params['data'], $params['raw_output'] );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params','rv'),true), E_USER_NOTICE); }
        return $rv;
    }


    /**
     * get array of valid, local realpaths from string|array of path(s)
	 * @param array|string $input   : paths to check
     * @return array                : array of valid realpaths
     */
    public static function getRealPaths( $input )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('input'),true), E_USER_NOTICE); }

        $delim = '|';
        $valid = 'A-z0-9-_+\.\/:"*?<>"=';
        // to string
        if (is_array($input))
            { $input = join($delim, $input); }
        $input = preg_replace("|[^$valid]+|", $delim, $input);
        // to array
        $result = explode($delim, $input);
        foreach( $result as $key => $path )
        {
            $result[$key] = realpath($path);
            if (empty($path) || empty($result[$key]))
                { unset($result[$key]); }
        }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('input','result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * get array of local files under path
     * OR an XML list/fragment if format="xml":
     *     <file basename="{filename}" bytes="{filesize}">{filepath}</file>
     * final search is restricted to ABSPATH or search_paths
     *
     * @param string $path          : local directory path
     * @param string $match         : preg_match for full filepath
     * @param string $levels        : recursive
     * @param array $search_paths   :
     * @param array $format         : php (dflt) | xml
     * @return mixed                : array or XML with valid realpaths
     */
    public static function getFileListingLocal( $path, $match = '.xml$', $levels = 1, $search_paths = array(), $format = 'php' )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('path','match','levels','search_paths','format'),true), E_USER_NOTICE); }

        if (strpos($path,'__') !== false)
        {
            $search  = array('__WP_HOME_DIR__', '__WP_CONTENT_DIR__', '__IMG_PROCESSOR_DIR__');
            $replace = array( ABSPATH,           WP_CONTENT_DIR,       IMG_PROCESSOR_DIR);
            $path = str_replace($search, $replace, $path);
        }
        if (!empty($search_paths) && !is_array($search_paths))
            { $search_paths = array($search_paths); }

        $dir_path = realpath($path);
        $valid_path = $dir_path && is_dir($dir_path) && is_readable($dir_path);
        if ($valid_path && strpos($dir_path,ABSPATH) === false)
        {
            $valid_path = false;
            foreach( $search_paths as $search_path )
            {
                if (strpos($dir_path,$search_path) === false)
                    { continue; }
                $valid_path = true;
                break;
            }
        }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('path','dir_path','valid_path'),true), E_USER_NOTICE); }

        $result = array();
        if ($valid_path)
        {
//if (WP_DEBUG) { trigger_error(__METHOD__." : OPENDIR( $dir_path )", E_USER_NOTICE); }
            if ($dir = opendir($dir_path))
            {
                while (($file = readdir($dir)) !== false) {
                    $fullpath = rtrim($dir_path,'/').'/'.$file;

                    if ('.' === $file[0] || !is_readable($fullpath))
                        { continue; }
                    if (is_dir($fullpath) && $levels > 1)
                    {
                        $subresult = self::getFileListingLocal( $fullpath, $match, $levels-1 );
                        if (is_array($subresult))
                            { $result = array_merge( $subresult, $result ); }
                        continue;
                    }
                    if (!pathinfo($file, PATHINFO_EXTENSION))
                        { continue; }
                    if ($match && !preg_match('/'.$match.'/i', $fullpath))
                        { continue; }

                    $result[] = $fullpath;
                }
                closedir($dir);
            }
        }
        else
        {
            foreach( $search_paths as $search_path )
            {
                if (empty($search_path)) { continue; }
                $fullpath = rtrim($search_path,'/').'/'.$path;

                if (!($fullpath && is_dir($fullpath) && is_readable($fullpath)))
                    { continue; }

                $subresult = self::getFileListingLocal( $fullpath, $match, $levels );
                if (is_array($subresult))
                    { $result = array_merge( $subresult, $result ); }
            }
        }
        if (!empty($result))
        {
            $result = array_unique( $result, SORT_STRING );
            sort( $result, SORT_STRING );
        }

        if ($format != 'xml')
        {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('path','match','levels','result'),true), E_USER_NOTICE); }
            return $result;
        }

        $rv = '';
        foreach ($result as $file) {
            $basename = pathinfo($file,  PATHINFO_BASENAME);
            $bytes    = filesize($file);
            $rv .= '<file'
                . ' basename="'.$basename.'"'
                . ' bytes="'.$bytes.'"'
                .'>';
            $rv .= html_entity_decode($file,ENT_XML1,"UTF-8");
            $rv .= '</file>';
        }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('path','match','levels','rv'),true), E_USER_NOTICE); }
        return $rv;
    }

    /**
     * check local file exists
     * __WP_HOME_DIR__, __WP_CONTENT_DIR__ and __IMG_PROCESSOR_DIR__ automatically replaced
     *
     * @see file.xsl, template name="file-exists-local"
     * @param string $file          : local path passed to realpath()
     * @param array $search_paths   :
     * @return string               : realpath or empty
     */
    public static function getFileExistsLocal( $file, $search_paths = array() )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('file','search_paths'),true), E_USER_NOTICE); }

        $file = trim($file);
        if (strpos($file,'__') !== false)
        {
            $search  = array('__WP_HOME_DIR__', '__WP_CONTENT_DIR__', '__IMG_PROCESSOR_DIR__');
            $replace = array( ABSPATH,           WP_CONTENT_DIR,       IMG_PROCESSOR_DIR);
            $file = str_replace($search, $replace, $file);
        }

        if (strpos($file,'/') === 0 && is_file($file))
        {
            return realpath( $file ) ?: $file;
        }

        if (!is_array($search_paths)) { $search_paths = array($search_paths); }
        foreach( $search_paths as $path )
        {
            if (empty($path)) { continue; }
            $fullpath = rtrim($path,'/').'/'.$file;
            if (is_file($fullpath))
            {
                return realpath( $fullpath ) ? realpath( $fullpath ) : $fullpath;
            }
        }
        return '';
    }

    /**
     * retrieve local file body
     *
     * @param string $file              : local path passed to file_get_contents()
     * @param integer $cache_minutes    : cache_minutes for wp transient
     * @return string                   : page body
     */
    public static function getLocalFile( $file, $cache_minutes = 1 )
    {
        if (is_file($file) && is_readable($file))
        {
            return file_get_contents( $file );
        }
        return '';
    }


    /**
     * check remote file exists
     *
     * @see file.xsl, template name="file-exists-remote"
     * @param string $url   : remote path passed to cURL
     * @return string       : url or empty
     */
    public static function getFileExistsRemote( $url )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('url'),true), E_USER_NOTICE); }

        $response = wp_remote_head( $url );
        if (is_wp_error( $response ))
        {
            $err = $response->get_error_message();
            trigger_error(__METHOD__." : wp_remote_head ERROR : ".print_r($err,true), E_USER_NOTICE);
            return print_r($err,true);
        }
        $response_code = wp_remote_retrieve_response_code( $response );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('url','response_code','response'),true), E_USER_NOTICE); }
        return ($response_code < 400) ? $url : '';
    }


    /**
     * retrieve remote file body
     *
     * @param string $url               : remote path passed to wp_remote_get()
     * @param integer $cache_minutes    : >0 for set_transient(), else delete_transient()
     * @return string                   : response body
     */
    public static function getRemoteFile( $url, $cache_minutes = 1 )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('url','cache_minutes'),true), E_USER_NOTICE); }

        $cache_params = array( 'method' => 'md5', 'data' => $url );
        $cache_key = self::getHash( $cache_params );
        $body = get_transient( $cache_key );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('url','cache_key','body'),true), E_USER_NOTICE); }

        if ($body !== false)
        {
            if ($cache_minutes > 0)
            {
if (WP_DEBUG) { trigger_error(__METHOD__." : CACHE GET : $cache_key", E_USER_NOTICE); }
                return $body;
            }
            else
            {
if (WP_DEBUG) { trigger_error(__METHOD__." : CACHE DELETE : $cache_key", E_USER_NOTICE); }
                delete_transient( $cache_key );
            }
        }

        $request_args = array(
            'timeout' => 30     // pref ???
            );
        $response = wp_remote_get( $url, $request_args );
        if (is_wp_error( $response ))
        {
            $err = $response->get_error_message();
            trigger_error(__METHOD__." : wp_remote_get ERROR : ".print_r($err,true), E_USER_NOTICE);
            return print_r($err,true);
        }
        $response_code = wp_remote_retrieve_response_code( $response );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('url','response_code','response'),true), E_USER_NOTICE); }

        $body = wp_remote_retrieve_body( $response );
        //$body = self::utf8_clean( $body );
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('url','response_code','body'),true), E_USER_NOTICE); }

        if (0 < $cache_minutes && $response_code < 400)
        {
if (WP_DEBUG) { trigger_error(__METHOD__." : CACHE SET : $cache_key", E_USER_NOTICE); }
            set_transient( $cache_key, $body, 60 * $cache_minutes );
        }
        return $body;
    }


    /**
     * call WP function shortcode_atts()
     * non-empty values starting with  '0|n|f' are set FALSE
     * remaining non-empty values are set TRUE
     *
     * @see https://developer.wordpress.org/reference/functions/shortcode_atts/
     */
    public static function getShortcodeBooleans( $pairs, $attrs, $shortcode = '' )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('pairs','attrs','shortcode'),true), E_USER_NOTICE); }
        $result = shortcode_atts( $pairs, $attrs, $shortcode );

        $no_vals = array( '0', 'n', 'f' );
        foreach( $result as $key => $val ) {
            $char1 = substr(strtolower(strval($val)), 0, 1);
            if (!empty($val) && in_array($char1, $no_vals))
                { $val = false; }
            if (!empty($val) || in_array($key,$attrs))
                { $val = true; }
             $result[$key] = $val;
        }
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('pairs','result'),true), E_USER_NOTICE); }
        return $result;
    }

    /**
     * Convert getImageCache() params to WP-CLI equivalents:
     *   'shortcode', 'command', 'nocache'
     *
     * @usedby IMG_Processor_GD::getImageCache()
     * @usedby IMG_Processor_Imagick::getImageCache()
     */
    public static function getImageCommands( $params, $info )
    {
//if (WP_DEBUG) { trigger_error(__METHOD__." : ".print_r(compact('params'),true), E_USER_NOTICE); }

        $transform = (empty($params['transform']['type']) || $params['transform']['type'] == 'none')
            ? 'info'
            : $params['transform']['type'];
        $file = strtr($params['input']['file'], array(ABSPATH => '/'));

        $shortcode = '[img_' .           $transform;
        $command   = 'wp img_process ' . $transform;
        $qry       = array(
            'source'    => $file,
            'transform' => $params['transform']['type'] ?? 'none',
            );

        $shortcode .= ' output="img"';
        $command   .= ' --output="info"';

        foreach( $params['transform'] as $key => $val ) {
            if ($key == 'type' || is_null($val)) { continue; }
            $shortcode .= ' '.$key.'="' .   addslashes($val).'"';
            $command   .= ' --'.$key.'="' . addslashes($val).'"';
            $qry[$key]  = $val;
        }
        foreach( $params['output'] as $key => $val ) {
            if (!in_array($key,array('type','dpi')) || is_null($val)) { continue; }
            $key = 'output_'.$key;
            $shortcode .= ' '.$key.'="' .   addslashes($val).'"';
            $command   .= ' --'.$key.'="' . addslashes($val).'"';
            $qry[$key]  = $val;
        }

        $shortcode  .= ']'.$file.'[/img_'.$transform.']';
        $command    .= ' '.$file;
        $qry['output_nocache'] = 1;

        $query   = IMG_PROCESSOR_URL.'?'.http_build_query($qry);
        $nocache = (!empty($info['img']))
            ? preg_replace('|src="[^\"]+"|', 'src="'.$query.'"', $info['img'])
            : '<img src="' . $query . '" width="'.$info['width'].'" height="'.$info['height'].'" />';

        $sc_figure  = '';
        $sc_figure .= ' figure-id="FIGURE-ID"';
        $sc_figure .= ' figure-class="FIGURE-CLASS"';
        $sc_figure .= ' caption="CAPTION"';
        $sc_figure .= ' caption-class="CAPTION-CLASS"';

        $sc_html  = '';
        $sc_html .= ' img-id="IMG-ID"';
        $sc_html .= ' alt="ALT"';
        $sc_html .= ' title="TITLE"';
        $sc_html .= ' style="STYLE"';
        $sc_html .= ' decoding="auto|async|sync"';
        $sc_html .= ' loading="eager|lazy"';

        $sc_figure = 'output="figure"' . $sc_figure . $sc_html;
        $sc_html   = 'output="html"' . $sc_html;

        $result = array(
            'cli-command'  => $command,
            'api-nocache'  => $nocache,
            'wp-shortcode-img'    => $shortcode,
            'wp-shortcode-html'   => preg_replace("|output=\"[a-z]+\"|", $sc_html,   $shortcode),
            'wp-shortcode-figure' => preg_replace("|output=\"[a-z]+\"|", $sc_figure, $shortcode),
            );
        return $result;
    }


} // end class IMG_Processor_Util
